const { Telegraf, Markup } = require('telegraf');
const axios = require('axios');
require('dotenv').config();

const bot = new Telegraf(process.env.TELEGRAM_BOT_TOKEN);
const API_URL = process.env.API_URL || 'http://pomo-api:3000/api';

// In-memory store for user tokens (in production, use Redis or DB)
const userTokens = {};

// Middleware to check auth
const authMiddleware = async (ctx, next) => {
  const chatId = ctx.chat.id.toString();
  if (!userTokens[chatId]) {
    return ctx.reply('⚠️ You are not logged in. Please use /login <email> <password> to authenticate.');
  }
  ctx.apiToken = userTokens[chatId];
  return next();
};

bot.start((ctx) => {
  ctx.reply(
    `🍅 Welcome to Pomo Bot!\n\n` +
    `I can help you manage your Pomodoro tasks.\n\n` +
    `Commands:\n` +
    `/login <email> <password> - Authenticate with Pomo API\n` +
    `/tasks - List your active tasks\n` +
    `/add <message> - Add a new task\n` +
    `/stats - View your productivity stats\n` +
    `/logout - Clear your session`
  );
});

bot.command('login', async (ctx) => {
  const args = ctx.message.text.split(' ');
  if (args.length !== 3) {
    return ctx.reply('Usage: /login <email> <password>');
  }

  const [_, email, password] = args;

  try {
    const response = await axios.post(`${API_URL}/auth/login`, {
      email,
      password
    });

    const { token, user } = response.data;
    userTokens[ctx.chat.id.toString()] = token;

    ctx.reply(`✅ Logged in successfully as ${user.username}!`);
  } catch (error) {
    console.error('Login error:', error.response?.data || error.message);
    ctx.reply('❌ Login failed. Please check your credentials.');
  }
});

bot.command('logout', (ctx) => {
  delete userTokens[ctx.chat.id.toString()];
  ctx.reply('👋 Logged out successfully.');
});

// Protected commands
bot.command('tasks', authMiddleware, async (ctx) => {
  try {
    const response = await axios.get(`${API_URL}/tasks`, {
      headers: { Authorization: `Bearer ${ctx.apiToken}` }
    });

    const tasks = response.data;
    if (tasks.length === 0) {
      return ctx.reply('📝 You have no tasks yet. Use /add to create one.');
    }

    let message = '📋 **Your Tasks:**\n\n';
    tasks.forEach(task => {
      const statusEmoji = task.status === 'completed' ? '✅' : '⏳';
      message += `${statusEmoji} *${task.message}*\n`;
      message += `   🍅 ${task.pomodoros} pomodoros | ⏱ ${task.duration}\n\n`;
    });

    ctx.replyWithMarkdown(message);
  } catch (error) {
    console.error('Tasks error:', error.message);
    ctx.reply('❌ Failed to fetch tasks.');
  }
});

bot.command('add', authMiddleware, async (ctx) => {
  const message = ctx.message.text.replace('/add', '').trim();
  if (!message) {
    return ctx.reply('Usage: /add <task description>');
  }

  try {
    const response = await axios.post(`${API_URL}/tasks`, {
      message: message,
      pomodoros: 1,
      duration: '25m'
    }, {
      headers: { Authorization: `Bearer ${ctx.apiToken}` }
    });

    ctx.reply(`✅ Task added: "${response.data.message}"`);
  } catch (error) {
    console.error('Add task error:', error.message);
    ctx.reply('❌ Failed to add task.');
  }
});

bot.command('stats', authMiddleware, async (ctx) => {
  try {
    const response = await axios.get(`${API_URL}/stats`, {
      headers: { Authorization: `Bearer ${ctx.apiToken}` }
    });

    const stats = response.data;
    ctx.reply(
      `📊 **Productivity Stats**\n\n` +
      `Total Tasks: ${stats.totalTasks}\n` +
      `Completed Sessions: ${stats.completedSessions}\n` +
      `Total Focus Time: ${Math.round(stats.totalPomodoroTime / 60)} mins`
    );
  } catch (error) {
    console.error('Stats error:', error.message);
    ctx.reply('❌ Failed to fetch stats.');
  }
});

bot.launch();
console.log('🤖 Pomo Bot is running...');

// Enable graceful stop
process.once('SIGINT', () => bot.stop('SIGINT'));
process.once('SIGTERM', () => bot.stop('SIGTERM'));
