# Pomo API

A RESTful API backend for managing Pomodoro tasks with real-time updates.

## Features

- ✅ RESTful API for task management
- ✅ PostgreSQL database for scalable storage
- ✅ Real-time updates via WebSockets
- ✅ Authentication system (JWT)
- ✅ Rate limiting and security headers
- ✅ Statistics and analytics
- ✅ Migration from existing SQLite data

## Quick Start

### Prerequisites

1. **Node.js** (v16 or higher)
2. **PostgreSQL** (v12 or higher)

### Installation

1. Clone the repository:
```bash
git clone <your-repo-url>
cd pomo-api
```

2. Install dependencies:
```bash
npm install
```

3. Set up environment variables:
```bash
cp .env.example .env
# Edit .env with your database credentials
```

4. Set up PostgreSQL database:
```bash
# Create database
createdb pomo_db

# Run migrations
npm run migrate
```

5. Start the server:
```bash
# Development
npm run dev

# Production
npm start
```

## API Endpoints

### Tasks
- `GET /api/tasks` - Get all tasks
- `POST /api/tasks` - Create new task
- `PUT /api/tasks/:id` - Update task
- `DELETE /api/tasks/:id` - Delete task

### Pomodoro Sessions
- `GET /api/tasks/:id/sessions` - Get sessions for a task
- `POST /api/tasks/:id/sessions` - Start new session
- `PUT /api/sessions/:id` - Complete/update session

### Statistics
- `GET /api/stats` - Get productivity statistics

### Health Check
- `GET /api/health` - Server health status

## Database Schema

### Users Table
```sql
users (
  id SERIAL PRIMARY KEY,
  username VARCHAR(50) UNIQUE NOT NULL,
  email VARCHAR(100) UNIQUE NOT NULL,
  password_hash VARCHAR(255) NOT NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
)
```

### Tasks Table
```sql
tasks (
  id SERIAL PRIMARY KEY,
  user_id INTEGER REFERENCES users(id),
  message TEXT NOT NULL,
  pomodoros INTEGER DEFAULT 1,
  duration VARCHAR(20) DEFAULT '25m',
  tags TEXT DEFAULT '',
  status VARCHAR(20) DEFAULT 'active',
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
)
```

### Pomodoro Sessions Table
```sql
pomodoro_sessions (
  id SERIAL PRIMARY KEY,
  task_id INTEGER REFERENCES tasks(id),
  start_time TIMESTAMP NOT NULL,
  end_time TIMESTAMP,
  duration INTEGER NOT NULL,
  completed BOOLEAN DEFAULT FALSE,
  notes TEXT,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
)
```

## Real-time Updates

The API uses Socket.IO for real-time updates:

- `task_created` - New task created
- `task_updated` - Task modified
- `task_deleted` - Task removed
- `session_started` - Pomodoro session started
- `session_completed` - Pomodoro session finished

## Environment Variables

```env
PORT=3000
NODE_ENV=development
JWT_SECRET=your-super-secret-jwt-key
DATABASE_URL=postgresql://username:password@localhost:5432/pomo_db
CORS_ORIGIN=http://localhost:3001
```

## Migration from Local SQLite

To migrate your existing local pomo data:

1. Update the `DATABASE_URL` in `.env`
2. Run the migration script:
```bash
node migrate.js
```

This will:
- Create the database schema
- Import your existing tasks from the local SQLite database
- Create a default user account

## Deployment

### Railway
1. Connect your GitHub repository to Railway
2. Set the `DATABASE_URL` environment variable
3. Railway will automatically deploy

### Docker
```bash
docker build -t pomo-api .
docker run -p 3000:3000 --env-file .env pomo-api
```

## Security Features

- Rate limiting (100 requests per 15 minutes)
- Helmet.js security headers
- CORS configuration
- JWT authentication (to be fully implemented)
- Input validation

## Development

### Running Tests
```bash
npm test
```

### Database Migrations
```bash
npm run migrate
```

### API Testing
Use the health check endpoint:
```bash
curl http://localhost:3000/api/health
```

## Next Steps

1. ✅ Basic API structure
2. ✅ Database schema
3. ✅ Real-time updates
4. 🔄 User authentication (in progress)
5. 🔄 Mobile app API endpoints
6. 🔄 MCP integration points
7. 🔄 Web dashboard

## Contributing

1. Fork the repository
2. Create a feature branch
3. Make your changes
4. Add tests
5. Submit a pull request

## License

MIT License