import { ForumQuestion, ForumAnswer } from '../types/forum';

// FAQ data extracted from components/FAQ.tsx
const faqData = [
  {
    question: "Is knowbuild a SaaS product or available as outright purchase?",
    answer: "knowbuild is a SaaS product. We do not offer an outright purchase option because our team works continuously to improve the platform with new features, updates, and performance enhancements — all of which become available to you at no additional cost under the SaaS model. This ensures your system keeps improving every month without expensive upgrades or migrations. This is why we don't offer an outright purchase option.",
    module: "General"
  },
  {
    question: "Why is knowbuild called a QERP.CRM?",
    answer: "QERP stands for \"Quasi-ERP\" — an ERP-level operating system designed specifically for SMEs, with CRM built in at the core. Unlike traditional ERPs built for large enterprises, QERP is: lightweight yet powerful, fast to implement, built around the inquiry-to-cash workflow, and integrated deeply with CRM, operations, dispatch & finance. \"QERP.CRM\" simply means: A connected ERP + CRM system tailor-made for real-world SME teams, that provides ERP + CRM + automation + dispatch + intelligence — exactly what \"Quasi-ERP\" suggests.",
    module: "General"
  },
  {
    question: "How is knowbuild different from generic ERPs?",
    answer: "Generic ERPs are often bloated and hard to use. knowbuild is purpose-built exclusively for B2B SMEs, focusing specifically on the inquiry-to-cash cycle. It connects sales directly to operations and finance, systemizing your company from Day 1 — without the complexity of large enterprise systems and light-years ahead of basic accounting tools or generic CRM/ERP software in the market.",
    module: "General"
  },
  {
    question: "How long does implementation take?",
    answer: "For a typical SME, we aim for a 2-4 week Go-Live. Our team helps migrate your customer and product data so you can start quoting and billing quickly.",
    module: "General"
  },
  {
    question: "Can we start with just CRM and add operations later?",
    answer: "Yes. While knowbuild is a unified system, you can roll it out in phases. Start by fixing your sales funnel, then unlock procurement and dispatch when the team is ready.",
    module: "Sales"
  },
  {
    question: "Does it integrate with accounting software?",
    answer: "knowbuild handles all invoicing and receivables management. We provide exportable reports compatible with major accounting tools like Tally or QuickBooks for final ledger posting.",
    module: "Finance"
  }
];

// Existing mock questions from QuestionsPageContent
const existingMockQuestions = [
  {
    question: "How does the smart inquiry assignment work?",
    answer: "The smart inquiry assignment uses rules-based routing to automatically assign inquiries to the right sales rep based on region, product expertise, or workload. This ensures faster response times and better lead conversion.",
    module: "Inbound Sales",
    tags: ["assignment", "automation"]
  },
  {
    question: "Can we track accounts receivable by customer?",
    answer: "Yes, knowbuild provides detailed accounts receivable tracking with customer-wise ageing reports, payment history, and credit limit monitoring.",
    module: "Finance",
    tags: ["receivables", "payments"]
  },
  {
    question: "Does the dispatch module support barcode scanning?",
    answer: "Yes, the dispatch module includes scan-to-pack validation using barcode/QR code scanning to ensure 100% accuracy in shipments.",
    module: "Dispatch",
    tags: ["scanning", "accuracy"]
  }
];

/**
 * Convert FAQ data to ForumQuestion format
 */
function convertFAQToForumQuestion(faq: typeof faqData[0], index: number): ForumQuestion {
  const now = new Date().toISOString();
  return {
    id: `faq-${index + 1}`,
    question: faq.question,
    answer: faq.answer,
    module: faq.module,
    tags: [],
    status: 'answered' as const,
    createdAt: now,
    updatedAt: now,
    answerCount: 1
  };
}

/**
 * Convert existing mock questions to ForumQuestion format
 */
function convertMockToForumQuestion(mock: typeof existingMockQuestions[0], index: number): ForumQuestion {
  const now = new Date().toISOString();
  return {
    id: `mock-${index + 1}`,
    question: mock.question,
    answer: mock.answer,
    module: mock.module,
    tags: mock.tags || [],
    status: 'answered' as const,
    createdAt: now,
    updatedAt: now,
    answerCount: 1
  };
}

/**
 * Get initial forum questions (merged FAQ + existing mock questions)
 */
export function getInitialForumQuestions(): ForumQuestion[] {
  const faqQuestions = faqData.map((faq, index) => convertFAQToForumQuestion(faq, index));
  const mockQuestions = existingMockQuestions.map((mock, index) => convertMockToForumQuestion(mock, index));
  return [...faqQuestions, ...mockQuestions];
}

/**
 * Get initial forum answers for FAQ questions
 */
export function getInitialForumAnswers(): ForumAnswer[] {
  const answers: ForumAnswer[] = [];
  const questions = getInitialForumQuestions();
  
  questions.forEach((question) => {
    if (question.answer) {
      answers.push({
        id: `answer-${question.id}`,
        questionId: question.id,
        answer: question.answer,
        author: 'knowbuild Team',
        createdAt: question.createdAt,
        isOfficial: true
      });
    }
  });
  
  return answers;
}

