'use client';

import React from 'react';
import { motion } from 'framer-motion';

interface ButtonProps extends React.ButtonHTMLAttributes<HTMLButtonElement> {
  variant?: 'primary' | 'secondary' | 'ghost' | 'outline';
  size?: 'sm' | 'md' | 'lg';
  fullWidth?: boolean;
  icon?: React.ReactNode;
}

export const Button: React.FC<ButtonProps> = ({ 
  children, 
  variant = 'primary', 
  size = 'md', 
  fullWidth = false,
  icon,
  className = '',
  ...props 
}) => {
  const baseStyles =
    // NOTE: overflow is intentionally NOT hidden so dark-mode outer glows can extend beyond the button.
    "relative inline-flex items-center justify-center rounded-xl font-bold transition-all duration-300 focus:outline-none focus:ring-2 focus:ring-brand-blue/40 focus:ring-offset-0 disabled:opacity-50 disabled:cursor-not-allowed tracking-wide group";
  
  const variants = {
    // Primary: Brand Orange with subtle inner glow and shadow
    primary:
      "bg-brand-orange text-white shadow-glow-orange hover:bg-brand-lightOrange border border-white/20 " +
      // Dark mode (inspired by your snippet): outer glow + neon border + gradient body + inner top shine
      "dark:border-2 dark:border-orange-400/80 dark:bg-gradient-to-br dark:from-orange-500 dark:to-orange-700 " +
      "dark:shadow-[inset_0_2px_4px_rgba(255,255,255,0.30),0_0_15px_rgba(249,115,22,0.40)] " +
      "dark:hover:border-orange-300 dark:hover:from-orange-400 dark:hover:to-orange-600 " +
      "dark:hover:shadow-[inset_0_2px_6px_rgba(255,255,255,0.40),0_0_25px_rgba(249,115,22,0.60)] " +
      // Inner top shine
      "dark:before:content-[''] dark:before:absolute dark:before:inset-0 dark:before:rounded-[inherit] dark:before:pointer-events-none " +
      "dark:before:bg-gradient-to-b dark:before:from-white/30 dark:before:via-transparent dark:before:to-transparent " +
      // Soft outer glow (must be outside and behind the button)
      "dark:after:content-[''] dark:after:absolute dark:after:-inset-2 dark:after:rounded-[inherit] dark:after:pointer-events-none dark:after:-z-10 " +
      "dark:after:bg-orange-500/30 dark:after:blur-2xl dark:after:transition-all dark:after:duration-500 " +
      "dark:hover:after:bg-orange-500/50 dark:hover:after:blur-3xl " +
      // Icon pop
      "dark:[&>span>span>svg]:drop-shadow-[0_0_8px_rgba(255,255,255,0.45)]",
    // Secondary: Brand Blue with depth
    secondary: "bg-brand-blue text-white shadow-glow hover:bg-brand-lightBlue border border-white/20",
    ghost: "bg-transparent text-slate-600 hover:text-brand-blue hover:bg-brand-blue/5",
    outline:
      "border border-slate-200 text-slate-700 hover:border-brand-blue hover:text-brand-blue bg-white/50 backdrop-blur-sm " +
      // Dark mode (inspired by your snippet): outer glow + neon border + glassy fill + inner top shine
      "dark:border-2 dark:border-blue-400/80 dark:bg-blue-950/60 dark:text-blue-100 dark:backdrop-blur-sm " +
      "dark:shadow-[inset_0_0_10px_rgba(59,130,246,0.30),0_0_15px_rgba(59,130,246,0.40)] " +
      "dark:hover:border-blue-300 dark:hover:bg-blue-900/70 dark:hover:text-white " +
      "dark:hover:shadow-[inset_0_0_15px_rgba(59,130,246,0.50),0_0_25px_rgba(59,130,246,0.60)] " +
      // Inner top shine
      "dark:before:content-[''] dark:before:absolute dark:before:inset-0 dark:before:rounded-[inherit] dark:before:pointer-events-none " +
      "dark:before:bg-gradient-to-b dark:before:from-blue-300/20 dark:before:via-transparent dark:before:to-transparent " +
      // Soft outer glow (must be outside and behind the button)
      "dark:after:content-[''] dark:after:absolute dark:after:-inset-2 dark:after:rounded-[inherit] dark:after:pointer-events-none dark:after:-z-10 " +
      "dark:after:bg-blue-500/30 dark:after:blur-2xl dark:after:transition-all dark:after:duration-500 " +
      "dark:hover:after:bg-blue-500/50 dark:hover:after:blur-3xl " +
      // Icon pop
      "dark:[&>span>span>svg]:drop-shadow-[0_0_8px_rgba(59,130,246,0.80)]",
  };

  const sizes = {
    sm: "px-4 py-2 text-sm",
    md: "px-6 py-3 text-base",
    lg: "px-8 py-4 text-lg",
  };

  return (
    <motion.button 
      whileHover={{ scale: 1.02, y: -2 }}
      whileTap={{ scale: 0.95, y: 0 }}
      transition={{ type: "spring", stiffness: 400, damping: 17 }}
      className={`${baseStyles} ${variants[variant]} ${sizes[size]} ${fullWidth ? 'w-full' : ''} ${className}`}
      {...props as any}
    >
      {/* Shine effect on hover */}
      {/* We clip the shimmer only (not the whole button) so outer glows can still extend in dark mode. */}
      <div className="absolute inset-0 overflow-hidden rounded-[inherit] pointer-events-none z-10">
        <div className="absolute inset-0 -translate-x-full group-hover:animate-[shimmer_2s_infinite] bg-gradient-to-r from-transparent via-white/20 to-transparent dark:via-white/10" />
      </div>
      
      <span className="relative z-20 flex items-center gap-2">
        {icon && <span>{icon}</span>}
        {children}
      </span>
    </motion.button>
  );
};
