"use client";

import React from "react";
import { motion } from "framer-motion";
import {
  Calendar,
  Mail,
  ChevronDown,
  Target,
  DollarSign,
  ShoppingCart,
  Lightbulb,
  TrendingUp,
  FileText,
  Wallet,
  Tag,
  Package,
  BarChart,
} from "lucide-react";
import { Button } from "./ui/Button";

// Internal UI Components
const Card = ({ className, children }: { className?: string, children: React.ReactNode }) => (
  <div className={`bg-white dark:bg-slate-800 rounded-xl border border-slate-200 dark:border-slate-700 shadow-sm ${className}`}>
    {children}
  </div>
);

const Badge = ({ className, children }: { className?: string, children: React.ReactNode }) => (
  <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${className}`}>
    {children}
  </span>
);

const Progress = ({ value, className }: { value: number, className?: string }) => (
  <div className={`relative w-full overflow-hidden rounded-full bg-slate-100 dark:bg-slate-700 ${className}`}>
    <div className="h-full bg-slate-900 dark:bg-slate-100 transition-all" style={{ width: `${value}%` }} />
  </div>
);

// Mock data for tasks
const tasksData = [
  {
    time: "12:30 PM",
    title: "Local visit",
    name: "John Lee",
    email: "john.lee@gmail.com",
    phone: "88883948730",
    status: "pending",
    color: "#eb6a6a",
  },
  {
    time: "1:00 PM",
    title: "Out side visit",
    name: "Jofin George",
    email: "jofin.george@gmail.com",
    phone: "88883948730",
    status: "completed",
    color: "#1f9b34",
  },
  {
    time: "1:30 PM",
    title: "Reporting",
    name: "Hubert Wilson",
    email: "jofin.george@gmail.com",
    phone: "88883948730",
    status: "pending",
    color: "#eb6a6a",
  },
  {
    time: "1:30 PM",
    title: "Holiday",
    status: "pending",
    color: "#eb6a6a",
  },
];

// Mock data for bottom badges
const badgesData = [
  { label: "Escalations", count: 12, bg: "#d64343", icon: TrendingUp },
  {
    label: "Inquiries",
    count: 12,
    bg: "linear-gradient(to right, #eba24d, #ef8911)",
    icon: FileText,
  },
  { label: "Opportunities", count: 12, bg: "#3375d6", icon: Lightbulb },
  { label: "Payments", count: 12, bg: "#03a580", icon: Wallet },
  { label: "Open tender", count: 12, bg: "#683505", icon: Tag },
  { label: "Price updates", count: 8, bg: "#64257c", icon: Package },
  { label: "Trends", count: 12, bg: "#307a24", icon: BarChart },
];

// Stats for monthly target
const monthlyStats = [
  { label: "Offer", value: "01", bg: "#f6efd4", color: "#a28212", darkBg: "#423a18", darkColor: "#ffd54f" },

  { label: "Confirmed", value: "02", bg: "#ebf0fd", color: "#3c64ea", darkBg: "#1a2b5e", darkColor: "#7ca1ff" },
  { label: "Closed", value: "05", bg: "#d1eedb", color: "#19a94a", darkBg: "#0d3d1e", darkColor: "#4ade80" },
];

export default function SalesDashboard() {
  const containerVariants = {
    hidden: { opacity: 0 },
    visible: {
      opacity: 1,
      transition: {
        staggerChildren: 0.1,
      },
    },
  };

  const itemVariants = {
    hidden: { opacity: 0, y: 20 },
    visible: {
      opacity: 1,
      y: 0,
      transition: {
        type: "spring" as const,
        stiffness: 300,
        damping: 24,
      },
    },
  };

  return (
    <motion.div
      initial="hidden"
      animate="visible"
      variants={containerVariants}
      className="min-h-auto bg-blend-saturation bg-background/80 backdrop-blur-3xl drop-shadow-2xl p-2 sm:p-4 lg:p-6 rounded-lg dark:bg-slate-900/80 max-w-full overflow-hidden"
    >
      {/* Overview Section */}
      <motion.div variants={itemVariants} className="mb-3 sm:mb-6">
        <div className="flex flex-col sm:flex-row sm:items-center justify-between mb-2 sm:mb-4 gap-2 sm:gap-3">
          <h2 className="text-base sm:text-lg md:text-xl font-bold text-[#1b1b1b] dark:text-white">
            Overview
          </h2>
          <div className="flex flex-col sm:flex-row gap-1.5 sm:gap-2 md:gap-4">
            <div className="bg-white dark:bg-slate-800 border border-[#dddddd] dark:border-slate-600 rounded-lg px-2 sm:px-3 py-1 sm:py-1.5 text-xs sm:text-sm text-[#bbbbbb] dark:text-slate-400 w-full sm:w-auto flex items-center justify-between min-w-[100px] sm:min-w-[120px]">
              <span className="truncate">John Doe</span>
              <ChevronDown className="w-3 h-3 sm:w-4 sm:h-4 shrink-0" />
            </div>
            <div className="bg-white dark:bg-slate-800 border border-[#dddddd] dark:border-slate-600 rounded-lg px-2 sm:px-3 py-1 sm:py-1.5 text-xs sm:text-sm text-[#bbbbbb] dark:text-slate-400 w-full sm:w-auto flex items-center justify-between min-w-[100px] sm:min-w-[120px]">
              <span className="truncate">2024-2025</span>
              <ChevronDown className="w-3 h-3 sm:w-4 sm:h-4 shrink-0" />
            </div>
          </div>
        </div>

        {/* Stats Cards */}
        <div className="grid grid-cols-2 sm:grid-cols-2 lg:grid-cols-4 gap-2 sm:gap-4 md:gap-6">
          {/* Achieved Card with Progress */}
          <motion.div
            variants={itemVariants}
            className="col-span-2 sm:col-span-2 lg:col-span-1"
          >
            <Card className="p-2 sm:p-3 md:p-4 bg-white dark:bg-slate-800 rounded-[16px] sm:rounded-[20px] shadow-[0px_2px_6.3px_0px_#dbe4ff] dark:shadow-none h-[110px] sm:h-[130px] md:h-[150px] relative overflow-visible">
              <div className="flex items-start justify-between mb-3 sm:mb-4 md:mb-6">
                <div className="flex flex-col gap-1 sm:gap-1.5">
                  <div className="w-6 h-6 sm:w-7 sm:h-7 md:w-8 md:h-8 rounded-md bg-[#d9f4ee] dark:bg-[#03a580]/20 flex items-center justify-center">
                    <Target className="w-3 h-3 sm:w-4 sm:h-4 md:w-5 md:h-5 text-[#03a580] dark:text-[#03a580]" />
                  </div>
                  <div>
                    <p className="text-sm sm:text-base md:text-lg lg:text-xl font-medium text-[#03a580] dark:text-[#03a580] leading-tight">
                      ₹36,000,300
                    </p>
                    <p className="text-xs sm:text-sm md:text-base text-[#767676] dark:text-slate-400">
                      Achieved
                    </p>
                  </div>
                </div>
              </div>
              <div className="absolute bottom-2 sm:bottom-3 md:bottom-4 left-2 sm:left-3 md:left-4 right-2 sm:right-3 md:right-4">
                <div className="relative">
                  <Progress value={70} className="h-1.5 sm:h-2 bg-[#1b1b1b]/20 dark:bg-white/10" />
                  <motion.div
                    initial={{ scale: 0 }}
                    animate={{ scale: 1 }}
                    transition={{ delay: 0.5, type: "spring" }}
                    className="absolute -top-1.5 sm:-top-2 bg-white dark:bg-slate-700 border border-[#e8e8e8] dark:border-slate-600 px-1.5 sm:px-2 py-0.5 rounded-lg text-[10px] sm:text-xs font-semibold text-[#484848] dark:text-slate-200"
                    style={{ left: "66%" }}
                  >
                    70%
                  </motion.div>
                </div>
              </div>
            </Card>
          </motion.div>

          {/* Account Receivables */}
          <motion.div variants={itemVariants}>
            <Card className="p-2 sm:p-3 md:p-4 bg-white dark:bg-slate-800 rounded-[16px] sm:rounded-[20px] shadow-[0px_2px_6.3px_0px_#dbe4ff] dark:shadow-none h-[110px] sm:h-[130px] md:h-[150px] flex flex-col justify-between">
              <div className="w-8 h-8 sm:w-10 sm:h-10 md:w-12 md:h-12 rounded-md bg-[#efe3f3] dark:bg-[#9444af]/20 flex items-center justify-center">
                <DollarSign className="w-4 h-4 sm:w-5 sm:h-5 md:w-6 md:h-6 text-[#9444af] dark:text-[#bd6dd6]" />
              </div>
              <div>
                <p className="text-sm sm:text-base md:text-lg lg:text-xl font-medium text-[#1b1b1b] dark:text-white leading-tight">
                  ₹40,55,200
                </p>
                <p className="text-xs sm:text-sm md:text-base text-[#767676] dark:text-slate-400">
                  Account receivables
                </p>
              </div>
            </Card>
          </motion.div>

          {/* Orders in Hand */}
          <motion.div variants={itemVariants}>
            <Card className="p-2 sm:p-3 md:p-4 bg-white dark:bg-slate-800 rounded-[16px] sm:rounded-[20px] shadow-[0px_2px_6.3px_0px_#dbe4ff] dark:shadow-none h-[110px] sm:h-[130px] md:h-[150px] flex flex-col justify-between">
              <div className="w-8 h-8 sm:w-10 sm:h-10 md:w-12 md:h-12 rounded-md bg-[#fff0e6] dark:bg-[#fe6c00]/20 flex items-center justify-center">
                <ShoppingCart className="w-4 h-4 sm:w-5 sm:h-5 md:w-6 md:h-6 text-[#fe6c00] dark:text-[#ff8a47]" />
              </div>
              <div>
                <p className="text-sm sm:text-base md:text-lg lg:text-xl font-medium text-[#1b1b1b] dark:text-white leading-tight">
                  ₹30,55,200
                </p>
                <p className="text-xs sm:text-sm md:text-base text-[#767676] dark:text-slate-400">
                  Orders in hand
                </p>
              </div>
            </Card>
          </motion.div>

          {/* Opportunities */}
          <motion.div variants={itemVariants}>
            <Card className="p-2 sm:p-3 md:p-4 bg-white dark:bg-slate-800 rounded-[16px] sm:rounded-[20px] shadow-[0px_2px_6.3px_0px_#dbe4ff] dark:shadow-none h-[110px] sm:h-[130px] md:h-[150px] flex flex-col justify-between">
              <div className="w-8 h-8 sm:w-10 sm:h-10 md:w-12 md:h-12 rounded-md bg-[#d2e5f6] dark:bg-[#207ed4]/20 flex items-center justify-center">
                <Lightbulb className="w-4 h-4 sm:w-5 sm:h-5 md:w-6 md:h-6 text-[#207ed4] dark:text-[#4da3f0]" />
              </div>
              <div>
                <p className="text-sm sm:text-base md:text-lg lg:text-xl font-medium text-[#1b1b1b] dark:text-white leading-tight">
                  ₹3,00,55,200
                </p>
                <p className="text-xs sm:text-sm md:text-base text-[#767676] dark:text-slate-400">
                  Opportunities
                </p>
              </div>
            </Card>
          </motion.div>
        </div>
      </motion.div>

      {/* Today's Tasks and Monthly Target Section */}
      <div className="grid grid-cols-1 xl:grid-cols-2 gap-2 sm:gap-4 md:gap-6 mb-2 sm:mb-4 md:mb-6">
        {/* Today's Tasks */}
        <motion.div variants={itemVariants}>
          <Card className="bg-white dark:bg-slate-800 rounded-[16px] sm:rounded-[20px] shadow-[0px_2px_16px_0px_rgba(0,82,204,0.04)] dark:shadow-none overflow-hidden h-full">
            {/* Header */}
            <div className="bg-[#dfe5ff] dark:bg-[#1a2b5e] p-2 sm:p-3 md:p-4 flex flex-col sm:flex-row sm:items-center gap-2 sm:gap-3 border-b border-[#e8e8e8] dark:border-slate-700">
              <div className="flex items-center gap-2 sm:gap-3 flex-1">
                <div className="w-6 h-6 sm:w-7 sm:h-7 md:w-8 md:h-8 bg-[#0052cc]/10 dark:bg-[#0052cc]/30 rounded-lg flex items-center justify-center">
                  <Calendar className="w-3 h-3 sm:w-4 sm:h-4 md:w-5 md:h-5 text-[#0052cc] dark:text-[#7ca1ff]" />
                </div>
                <h3 className="text-sm sm:text-base md:text-lg lg:text-xl font-normal text-[#1b1b1b] dark:text-white">
                  Today's Tasks
                </h3>
              </div>
              <div className="flex flex-col sm:flex-row gap-1.5 sm:gap-2 md:gap-3">
                <div className="bg-white dark:bg-slate-800 border border-[#dddddd] dark:border-slate-600 rounded-lg px-2 sm:px-3 py-0.5 sm:py-1 text-xs sm:text-sm text-[#bbbbbb] dark:text-slate-400 w-full sm:w-auto flex items-center gap-1.5 sm:gap-2">
                  <span>Pending (9)</span>
                  <ChevronDown className="w-2.5 h-2.5 sm:w-3 sm:h-3 shrink-0" />
                </div>
                <Button
                  variant="outline"
                  size="sm"
                  className="text-[#0052cc] dark:text-[#7ca1ff] border-[#0052cc]/50 dark:border-[#7ca1ff]/50 w-full sm:w-auto hover:bg-[#0052cc]/5 dark:hover:bg-[#7ca1ff]/10 text-xs sm:text-sm px-2 sm:px-3 py-1 sm:py-1.5"
                >
                  View all
                </Button>
              </div>
            </div>

            {/* Tasks List */}
            <div className="max-h-[200px] sm:max-h-[280px] md:max-h-[336px] overflow-y-auto">
              {tasksData.map((task, index) => (
                <motion.div
                  key={index}
                  initial={{ opacity: 0, x: -20 }}
                  animate={{ opacity: 1, x: 0 }}
                  transition={{ delay: 0.1 * index }}
                  className="p-2 sm:p-3 border-b border-[#e8e8e8] dark:border-slate-700 flex gap-3 sm:gap-4 hover:bg-gray-50 dark:hover:bg-slate-700/50 transition-colors"
                >
                  {/* Time Badge */}
                  <div
                    className="w-[36px] h-[36px] sm:w-[44px] sm:h-[44px] md:w-[54px] md:h-[54px] rounded-lg border border-[#e4e7f0] dark:border-slate-600 flex flex-col items-center justify-center text-center relative shrink-0"
                    style={{
                      boxShadow: `0px 3px 0px 0px ${task.color} inset`,
                    }}
                  >
                    <span className="text-[10px] sm:text-xs md:text-sm font-semibold text-[#1b1b1b] dark:text-white leading-tight">
                      {task.time.split(" ")[0]}
                    </span>
                    <span className="text-[8px] sm:text-[10px] md:text-xs text-[#767676] dark:text-slate-400 uppercase tracking-wider">
                      {task.time.split(" ")[1]}
                    </span>
                  </div>

                  {/* Task Details */}
                  <div className="flex-1 min-w-0">
                    <div className="flex items-center justify-between mb-0.5 sm:mb-1 gap-1 sm:gap-2">
                      <h4 className="text-xs sm:text-sm font-semibold text-[#1b1b1b] dark:text-white truncate">
                        {task.title}
                      </h4>
                      <div className="flex items-center gap-2 sm:gap-3 md:gap-4 shrink-0">
                        <Mail className="w-3 h-3 sm:w-4 sm:h-4 md:w-5 md:h-5 text-[#bbbbbb] dark:text-slate-500" />
                        <Badge
                          className={`${
                            task.status === "pending"
                              ? "bg-[#cd6c08]/10 dark:bg-[#cd6c08]/20 text-[#cd6c08] dark:text-[#ffa552]"
                              : "bg-[#1f9b34]/10 dark:bg-[#1f9b34]/20 text-[#1f9b34] dark:text-[#4ade80]"
                          } text-[10px] sm:text-xs uppercase px-1.5 sm:px-2 py-0.5`}
                        >
                          {task.status}
                        </Badge>
                      </div>
                    </div>
                    {task.name && (
                      <div className="flex flex-col sm:flex-row sm:items-center gap-1 sm:gap-2 text-xs sm:text-sm text-[#484848] dark:text-slate-300 mb-1.5">
                        <span>{task.name}</span>
                        <span className="hidden sm:inline text-[#dddddd] dark:text-slate-600">
                          |
                        </span>
                        <span className="truncate">{task.email}</span>
                        <span className="hidden sm:inline text-[#dddddd] dark:text-slate-600">
                          |
                        </span>
                        <span>{task.phone}</span>
                      </div>
                    )}
                    <div className="flex items-center gap-1.5 sm:gap-2 md:gap-3 flex-wrap">
                      <button className="text-[10px] sm:text-xs font-semibold text-[#0052cc] dark:text-[#7ca1ff] hover:underline">
                        View task
                      </button>
                      {task.name && (
                        <>
                          <span className="text-[#dddddd] dark:text-slate-600 hidden sm:inline">|</span>
                          <button className="text-[10px] sm:text-xs font-semibold text-[#0052cc] dark:text-[#7ca1ff] hover:underline">
                            View Inquiry
                          </button>
                        </>
                      )}
                    </div>
                  </div>
                </motion.div>
              ))}
            </div>
          </Card>
        </motion.div>

        {/* Monthly Target */}
        <motion.div variants={itemVariants}>
          <Card className="bg-white dark:bg-slate-800 rounded-[16px] sm:rounded-[20px] shadow-[0px_2px_16px_0px_rgba(0,82,204,0.04)] dark:shadow-none overflow-hidden h-full">
            {/* Header */}
            <div className="bg-[#f5e0c6] dark:bg-[#4a2308] p-2 sm:p-3 md:p-4 flex flex-col sm:flex-row sm:items-center gap-2 sm:gap-3 border-b border-[#e8e8e8] dark:border-slate-700">
              <div className="flex items-center gap-2 sm:gap-3 flex-1">
                <div className="w-6 h-6 sm:w-7 sm:h-7 md:w-8 md:h-8 bg-[#a28212]/10 dark:bg-[#a28212]/30 rounded-lg flex items-center justify-center">
                  <Target className="w-3 h-3 sm:w-4 sm:h-4 md:w-5 md:h-5 text-[#a28212] dark:text-[#ffd54f]" />
                </div>
                <h3 className="text-sm sm:text-base md:text-lg lg:text-xl font-normal text-[#1b1b1b] dark:text-white">
                  Monthly Target
                </h3>
              </div>
              <div className="flex flex-col sm:flex-row gap-1.5 sm:gap-2 md:gap-3">
                <div className="bg-white dark:bg-slate-800 border border-[#dddddd] dark:border-slate-600 rounded-lg px-2 sm:px-3 py-0.5 sm:py-1 text-xs sm:text-sm text-[#bbbbbb] dark:text-slate-400 w-full sm:w-auto flex items-center gap-1.5 sm:gap-2">
                  <span>March</span>
                  <ChevronDown className="w-2.5 h-2.5 sm:w-3 sm:h-3 shrink-0" />
                </div>
                <Button
                  variant="outline"
                  size="sm"
                  className="text-[#0052cc] dark:text-[#7ca1ff] border-[#0052cc]/50 dark:border-[#7ca1ff]/50 w-full sm:w-auto hover:bg-[#0052cc]/5 dark:hover:bg-[#7ca1ff]/10 text-xs sm:text-sm px-2 sm:px-3 py-1 sm:py-1.5"
                >
                  View list
                </Button>
              </div>
            </div>

            {/* Target Summary */}
            <div className="p-2 sm:p-3 md:p-6">
              <Card className="p-2 sm:p-3 md:p-4 bg-white dark:bg-slate-800 rounded-[16px] sm:rounded-[20px] shadow-[0px_2px_6.3px_0px_#dbe4ff] dark:shadow-none mb-2 sm:mb-4 md:mb-6">
                <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between gap-3 sm:gap-4 md:gap-6 lg:gap-10">
                  <div className="flex flex-col gap-3 sm:gap-4 md:gap-5">
                    <div className="w-8 h-8 sm:w-10 sm:h-10 md:w-12 md:h-12 rounded-md bg-[#f5e0c6] dark:bg-[#4a2308] flex items-center justify-center">
                      <Target className="w-4 h-4 sm:w-5 sm:h-5 md:w-6 md:h-6 text-[#a28212] dark:text-[#ffd54f]" />
                    </div>
                    <div>
                      <p className="text-sm sm:text-base md:text-lg lg:text-xl font-medium text-[#1b1b1b] dark:text-white mb-0.5 sm:mb-1 leading-tight">
                        ₹5,00,00,000
                      </p>
                      <p className="text-xs sm:text-sm md:text-base text-[#767676] dark:text-slate-400">
                        Sales Target For This Month
                      </p>
                    </div>
                  </div>
                  <div className="w-full h-px lg:w-px lg:h-20 md:lg:h-28 bg-[#e8e8e8] dark:bg-slate-700" />
                  <div className="flex flex-col gap-3 sm:gap-4 md:gap-5">
                    <div className="w-8 h-8 sm:w-10 sm:h-10 md:w-12 md:h-12 rounded-md bg-[#b3e8dc] dark:bg-[#0d3d1e] flex items-center justify-center lg:ml-auto">
                      <FileText className="w-4 h-4 sm:w-5 sm:h-5 md:w-6 md:h-6 text-[#03a580] dark:text-[#4ade80]" />
                    </div>
                    <div className="lg:text-right">
                      <p className="text-sm sm:text-base md:text-lg lg:text-xl font-medium text-[#1b1b1b] dark:text-white mb-0.5 sm:mb-1 leading-tight">
                        ₹3,20,00,300
                      </p>
                      <p className="text-xs sm:text-sm md:text-base text-[#767676] dark:text-slate-400">
                        Invoiced amount
                      </p>
                    </div>
                  </div>
                </div>
              </Card>

              {/* Stats Grid */}
              <div className="grid grid-cols-2 sm:grid-cols-3 gap-2 sm:gap-3">
                {monthlyStats.map((stat, index) => (
                  <div
                    key={index}
                    className="p-2 sm:p-3 md:p-4 rounded-[16px] sm:rounded-[20px] text-center transition-all"
                  >
                    <div
                      className="w-full h-full rounded-[16px] sm:rounded-[20px] p-2 sm:p-3 md:p-4 flex flex-col items-center justify-center relative overflow-hidden"
                      style={{
                        // Fallback for light mode (or default) if no dark class present
                        // We will use CSS variables for cleanest dynamic colors
                        "--card-bg": stat.bg,
                        "--card-color": stat.color,
                        "--card-dark-bg": stat.darkBg,
                        "--card-dark-color": stat.darkColor,
                      } as React.CSSProperties}
                    >
                       {/* Background layer controlled by Tailwind 'dark:' variants referring to CSS variables */}
                       <div className="absolute inset-0 bg-[var(--card-bg)] dark:bg-[var(--card-dark-bg)] z-0" />
                       
                       <div className="relative z-10 flex flex-col items-center text-[var(--card-color)] dark:text-[var(--card-dark-color)]">
                          <p className="text-sm sm:text-base md:text-lg lg:text-xl font-bold mb-0.5 sm:mb-1 leading-tight">
                            {stat.value}
                          </p>
                          <p className="text-[10px] sm:text-xs md:text-sm font-semibold opacity-80">
                            {stat.label}
                          </p>
                       </div>
                    </div>
                  </div>
                ))}
              </div>
            </div>
          </Card>
        </motion.div>
      </div>

      {/* Bottom Badges */}
      <motion.div
        variants={itemVariants}
        className="grid grid-cols-2 xs:grid-cols-2 sm:grid-cols-3 md:grid-cols-4 lg:grid-cols-5 xl:grid-cols-7 gap-1.5 sm:gap-2 md:gap-3 lg:gap-4"
      >
        {badgesData.map((badge, index) => (
          <motion.div
            key={index}
            initial={{ opacity: 0, scale: 0.8 }}
            animate={{ opacity: 1, scale: 1 }}
            transition={{ delay: 0.3 + index * 0.05 }}
            whileHover={{ scale: 1.05 }}
            className="rounded-full p-1.5 sm:p-2 md:p-3 flex items-center gap-1 sm:gap-1.5 md:gap-2 text-white cursor-pointer transition-all"
            style={{
              background: badge.bg,
              border: "1px solid rgba(255,255,255,0.3)",
            }}
          >
            <badge.icon className="w-3 h-3 sm:w-4 sm:h-4 md:w-5 md:h-5 shrink-0" />
            <span className="text-[10px] sm:text-xs md:text-sm flex-1 truncate">
              {badge.label}
            </span>
            <Badge className="bg-[#e8e8e8] dark:bg-slate-700 text-[#484848] dark:text-slate-200 text-[8px] sm:text-[10px] md:text-[11px] px-1 sm:px-1.5 py-0.5">
              {badge.count}
            </Badge>
          </motion.div>
        ))}
      </motion.div>
    </motion.div>
  );
}
