package main

import (
	"bytes"
	"context"
	"database/sql"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"log"
	"net/http"
	"os"
	"os/exec"
	"os/signal"
	"path/filepath"
	"sort"
	"strconv"
	"strings"
	"sync"
	"syscall"
	"time"

	"github.com/fatih/color"
	_ "github.com/mattn/go-sqlite3"
)

type Task struct {
	ID        int      `json:"id"`
	Message   string   `json:"message"`
	Pomodoros int      `json:"pomodoros"`
	Duration  string   `json:"duration"`
	Tags      []string `json:"tags"`
	Status    string   `json:"status"`
	CreatedAt string   `json:"created_at"`
	UpdatedAt string   `json:"updated_at"`
}

type PomodoroSession struct {
	ID        int    `json:"id"`
	TaskID    int    `json:"task_id"`
	StartTime string `json:"start_time"`
	EndTime   string `json:"end_time"`
	Duration  int    `json:"duration"`
	Completed bool   `json:"completed"`
	Notes     string `json:"notes"`
	CreatedAt string `json:"created_at"`
}

type LocalStore struct {
	dbPath     string
	apiURL     string
	authToken  string
	displayCfg DisplayConfig
}

const defaultDateTimeFormat = "2006-01-02 15:04"

type DisplayConfig struct {
	DateTimeFormat string
	TagColors      map[string]*color.Color
}

type TaskSummary struct {
	Task              Task
	Sessions          []PomodoroSession
	firstSessionStart *time.Time
	createdAtTime     *time.Time
}

type listOptions struct {
	jsonOutput    bool
	descending    bool
	showAll       bool
	limit         int
	durationRange string
}

func defaultListOptions() listOptions {
	return listOptions{
		showAll:       true,
		durationRange: "24h",
	}
}

func parseTimestamp(raw string) *time.Time {
	if raw == "" {
		return nil
	}

	layouts := []string{
		time.RFC3339Nano,
		time.RFC3339,
		"2006-01-02 15:04:05",
		"2006-01-02 15:04:05Z07:00",
	}

	for _, layout := range layouts {
		if t, err := time.Parse(layout, raw); err == nil {
			return &t
		}
	}

	return nil
}

func (ts TaskSummary) effectiveStart() *time.Time {
	if ts.firstSessionStart != nil {
		return ts.firstSessionStart
	}
	return ts.createdAtTime
}

type StatusState int

const (
	statusCreated StatusState = iota
	statusRunning
	statusBreaking
	statusComplete
	statusPaused
)

func (s StatusState) String() string {
	switch s {
	case statusCreated:
		return "CREATED"
	case statusRunning:
		return "RUNNING"
	case statusBreaking:
		return "BREAKING"
	case statusComplete:
		return "COMPLETE"
	case statusPaused:
		return "PAUSED"
	default:
		return ""
	}
}

type StatusInfo struct {
	TaskID        int           `json:"task_id"`
	TaskMessage   string        `json:"task_message"`
	State         StatusState   `json:"state"`
	Remaining     time.Duration `json:"remaining"`
	PauseDuration time.Duration `json:"pauseduration"`
	Count         int           `json:"count"`
	NPomodoros    int           `json:"n_pomodoros"`
}

type InteractiveTimer struct {
	store              *LocalStore
	session            *PomodoroSession
	task               *Task
	duration           time.Duration
	remaining          time.Duration
	state              StatusState
	paused             bool
	pauseStart         time.Time
	totalPaused        time.Duration
	mu                 sync.Mutex
	ctx                context.Context
	cancel             context.CancelFunc
	breakDuration      time.Duration
	inBreak            bool
	completedPomodoros int
}

func NewLocalStore(apiURL string) *LocalStore {
	dbPath := os.Getenv("POMO_DB_PATH")
	if dbPath == "" {
		home, err := os.UserHomeDir()
		if err != nil {
			log.Fatal("Error getting home directory:", err)
		}
		dbPath = filepath.Join(home, ".local", "share", "pomo", "pomo.db")
	} else {
		var err error
		dbPath, err = expandUserPath(dbPath)
		if err != nil {
			log.Fatal("Error expanding POMO_DB_PATH:", err)
		}
	}

	if err := os.MkdirAll(filepath.Dir(dbPath), 0o755); err != nil {
		log.Fatal("Error creating directory:", err)
	}

	displayCfg := loadDisplayConfig()

	// Read API token from environment
	apiToken := os.Getenv("POMO_API_TOKEN")

	return &LocalStore{dbPath: dbPath, apiURL: apiURL, authToken: apiToken, displayCfg: displayCfg}
}

func expandUserPath(path string) (string, error) {
	if path == "" {
		return "", fmt.Errorf("path is empty")
	}

	if strings.HasPrefix(path, "~") {
		home, err := os.UserHomeDir()
		if err != nil {
			return "", err
		}

		if path == "~" {
			return home, nil
		}

		if strings.HasPrefix(path, "~/") {
			return filepath.Join(home, path[2:]), nil
		}

		return "", fmt.Errorf("cannot expand user home in path: %s", path)
	}

	return path, nil
}

func loadDisplayConfig() DisplayConfig {
	cfg := DisplayConfig{
		DateTimeFormat: defaultDateTimeFormat,
		TagColors:      map[string]*color.Color{},
	}

	configPath := os.Getenv("POMO_CONFIG_PATH")
	if configPath == "" {
		home, err := os.UserHomeDir()
		if err == nil {
			configPath = filepath.Join(home, ".config", "pomo", "config.json")
		}
	}

	if configPath == "" {
		return cfg
	}

	raw, err := os.ReadFile(configPath)
	if err != nil {
		return cfg
	}

	type rawConfig struct {
		Colors      map[string]string `json:"colors"`
		DateTimeFmt string            `json:"dateTimeFmt"`
	}

	var parsed rawConfig
	if err := json.Unmarshal(raw, &parsed); err != nil {
		return cfg
	}

	if parsed.DateTimeFmt != "" {
		cfg.DateTimeFormat = parsed.DateTimeFmt
	}

	if len(parsed.Colors) > 0 {
		cfg.TagColors = buildColorMap(parsed.Colors)
	}

	return cfg
}

func buildColorMap(raw map[string]string) map[string]*color.Color {
	result := make(map[string]*color.Color)
	for tag, name := range raw {
		trimmed := strings.TrimSpace(tag)
		if trimmed == "" {
			continue
		}
		if c := lookupColor(name); c != nil {
			result[trimmed] = c
			result[strings.ToLower(trimmed)] = c
		}
	}
	return result
}

func lookupColor(name string) *color.Color {
	switch strings.ToLower(name) {
	case "black":
		return color.New(color.FgBlack)
	case "hiblack":
		return color.New(color.FgHiBlack)
	case "blue":
		return color.New(color.FgBlue)
	case "hiblue":
		return color.New(color.FgHiBlue)
	case "cyan":
		return color.New(color.FgCyan)
	case "hicyan":
		return color.New(color.FgHiCyan)
	case "green":
		return color.New(color.FgGreen)
	case "higreen":
		return color.New(color.FgHiGreen)
	case "magenta":
		return color.New(color.FgMagenta)
	case "himagenta":
		return color.New(color.FgHiMagenta)
	case "red":
		return color.New(color.FgRed)
	case "hired":
		return color.New(color.FgHiRed)
	case "white":
		return color.New(color.FgWhite)
	case "hiwhite", "hiwrite":
		return color.New(color.FgHiWhite)
	case "yellow":
		return color.New(color.FgYellow)
	case "hiyellow":
		return color.New(color.FgHiYellow)
	default:
		return nil
	}
}

func (ls *LocalStore) openDB() (*sql.DB, error) {
	return sql.Open("sqlite3", ls.dbPath)
}

func (ls *LocalStore) initDB() error {
	db, err := ls.openDB()
	if err != nil {
		return err
	}
	defer db.Close()

	_, err = db.Exec(`
		CREATE TABLE IF NOT EXISTS tasks (
			id INTEGER PRIMARY KEY AUTOINCREMENT,
			message TEXT NOT NULL,
			pomodoros INTEGER DEFAULT 1,
			duration VARCHAR(20) DEFAULT '25m',
			tags TEXT DEFAULT '',
			status VARCHAR(20) DEFAULT 'active',
			created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
			updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
		);

		CREATE TABLE IF NOT EXISTS pomodoro_sessions (
			id INTEGER PRIMARY KEY AUTOINCREMENT,
			task_id INTEGER REFERENCES tasks(id) ON DELETE CASCADE,
			start_time TIMESTAMP NOT NULL,
			end_time TIMESTAMP,
			duration INTEGER NOT NULL,
			completed BOOLEAN DEFAULT FALSE,
			notes TEXT,
			created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
		);
	`)
	return err
}

func (ls *LocalStore) createTask(task Task) (int, error) {
	db, err := ls.openDB()
	if err != nil {
		return 0, err
	}
	defer db.Close()

	tagsStr := strings.Join(task.Tags, ",")
	result, err := db.Exec(
		"INSERT INTO tasks (message, pomodoros, duration, tags, status) VALUES (?, ?, ?, ?, ?)",
		task.Message, task.Pomodoros, task.Duration, tagsStr, task.Status,
	)
	if err != nil {
		return 0, err
	}

	id, err := result.LastInsertId()
	return int(id), err
}

func (ls *LocalStore) getAllTasks() ([]Task, error) {
	db, err := ls.openDB()
	if err != nil {
		return nil, err
	}
	defer db.Close()

	rows, err := db.Query("SELECT id, message, pomodoros, duration, tags, status, created_at, updated_at FROM tasks ORDER BY created_at DESC")
	if err != nil {
		return nil, err
	}
	defer rows.Close()

	var tasks []Task
	for rows.Next() {
		var task Task
		var tagsStr string
		err := rows.Scan(&task.ID, &task.Message, &task.Pomodoros, &task.Duration, &tagsStr, &task.Status, &task.CreatedAt, &task.UpdatedAt)
		if err != nil {
			return nil, err
		}
		if tagsStr != "" {
			task.Tags = strings.Split(tagsStr, ",")
		}
		tasks = append(tasks, task)
	}
	return tasks, nil
}

func (ls *LocalStore) getTaskByID(taskID int) (*Task, error) {
	db, err := ls.openDB()
	if err != nil {
		return nil, err
	}
	defer db.Close()

	var task Task
	var tagsStr string
	err = db.QueryRow(
		"SELECT id, message, pomodoros, duration, tags, status, created_at, updated_at FROM tasks WHERE id = ?",
		taskID,
	).Scan(&task.ID, &task.Message, &task.Pomodoros, &task.Duration, &tagsStr, &task.Status, &task.CreatedAt, &task.UpdatedAt)
	if err == sql.ErrNoRows {
		return nil, nil
	}
	if err != nil {
		return nil, err
	}

	if tagsStr != "" {
		task.Tags = strings.Split(tagsStr, ",")
	}

	return &task, nil
}

func (ls *LocalStore) createSession(session PomodoroSession) (int, error) {
	db, err := ls.openDB()
	if err != nil {
		return 0, err
	}
	defer db.Close()

	result, err := db.Exec(
		"INSERT INTO pomodoro_sessions (task_id, start_time, duration, completed, notes) VALUES (?, ?, ?, ?, ?)",
		session.TaskID, session.StartTime, session.Duration, session.Completed, session.Notes,
	)
	if err != nil {
		return 0, err
	}

	id, err := result.LastInsertId()
	return int(id), err
}

func (ls *LocalStore) getTaskSessions(taskID int) ([]PomodoroSession, error) {
	db, err := ls.openDB()
	if err != nil {
		return nil, err
	}
	defer db.Close()

	rows, err := db.Query("SELECT id, task_id, start_time, end_time, duration, completed, notes, created_at FROM pomodoro_sessions WHERE task_id = ? ORDER BY start_time DESC", taskID)
	if err != nil {
		return nil, err
	}
	defer rows.Close()

	var sessions []PomodoroSession
	for rows.Next() {
		var session PomodoroSession
		var endTime sql.NullString
		err := rows.Scan(&session.ID, &session.TaskID, &session.StartTime, &endTime, &session.Duration, &session.Completed, &session.Notes, &session.CreatedAt)
		if err != nil {
			return nil, err
		}
		if endTime.Valid {
			session.EndTime = endTime.String
		}
		sessions = append(sessions, session)
	}
	return sessions, nil
}

func (ls *LocalStore) getSessionByID(sessionID int) (*PomodoroSession, error) {
	db, err := ls.openDB()
	if err != nil {
		return nil, err
	}
	defer db.Close()

	var session PomodoroSession
	var endTime sql.NullString
	err = db.QueryRow(
		"SELECT id, task_id, start_time, end_time, duration, completed, notes, created_at FROM pomodoro_sessions WHERE id = ?",
		sessionID,
	).Scan(&session.ID, &session.TaskID, &session.StartTime, &endTime, &session.Duration, &session.Completed, &session.Notes, &session.CreatedAt)
	if err == sql.ErrNoRows {
		return nil, nil
	}
	if err != nil {
		return nil, err
	}

	if endTime.Valid {
		session.EndTime = endTime.String
	}

	return &session, nil
}

func (ls *LocalStore) completeSession(sessionID int, notes string) error {
	session, err := ls.getSessionByID(sessionID)
	if err != nil {
		return err
	}
	if session == nil {
		return fmt.Errorf("session %d not found", sessionID)
	}
	if session.Completed {
		return fmt.Errorf("session %d is already marked complete", sessionID)
	}

	db, err := ls.openDB()
	if err != nil {
		return err
	}
	defer db.Close()

	end := time.Now().UTC()
	durationSeconds := session.Duration
	if start, err := time.Parse(time.RFC3339, session.StartTime); err == nil {
		if diff := end.Sub(start); diff > 0 {
			durationSeconds = int(diff.Seconds())
		}
	}

	_, err = db.Exec(
		`UPDATE pomodoro_sessions 
		 SET end_time = ?, completed = TRUE, duration = ?, notes = ? 
		 WHERE id = ?`,
		end.Format(time.RFC3339), durationSeconds, notes, sessionID,
	)
	if err != nil {
		return err
	}

	_, err = db.Exec(
		`UPDATE tasks SET updated_at = CURRENT_TIMESTAMP WHERE id = ?`,
		session.TaskID,
	)
	return err
}

func (ls *LocalStore) getTaskSessionsChronological(taskID int) ([]PomodoroSession, error) {
	db, err := ls.openDB()
	if err != nil {
		return nil, err
	}
	defer db.Close()

	rows, err := db.Query(
		"SELECT id, task_id, start_time, end_time, duration, completed, notes, created_at FROM pomodoro_sessions WHERE task_id = ? ORDER BY start_time ASC",
		taskID,
	)
	if err != nil {
		return nil, err
	}
	defer rows.Close()

	var sessions []PomodoroSession
	for rows.Next() {
		var session PomodoroSession
		var endTime sql.NullString
		err := rows.Scan(&session.ID, &session.TaskID, &session.StartTime, &endTime, &session.Duration, &session.Completed, &session.Notes, &session.CreatedAt)
		if err != nil {
			return nil, err
		}
		if endTime.Valid {
			session.EndTime = endTime.String
		}
		sessions = append(sessions, session)
	}
	return sessions, nil
}

func (ls *LocalStore) buildTaskSummaries(tasks []Task) ([]TaskSummary, error) {
	summaries := make([]TaskSummary, 0, len(tasks))
	for _, task := range tasks {
		sessions, err := ls.getTaskSessionsChronological(task.ID)
		if err != nil {
			return nil, err
		}
		summary := TaskSummary{
			Task:          task,
			Sessions:      sessions,
			createdAtTime: parseTimestamp(task.CreatedAt),
		}
		if len(sessions) > 0 {
			if start := parseTimestamp(sessions[0].StartTime); start != nil {
				summary.firstSessionStart = start
			}
		}
		summaries = append(summaries, summary)
	}
	return summaries, nil
}

func (ls *LocalStore) listTasks(opts listOptions) error {
	tasks, err := ls.getAllTasks()
	if err != nil {
		return err
	}

	summaries, err := ls.buildTaskSummaries(tasks)
	if err != nil {
		return err
	}

	if opts.descending {
		sort.SliceStable(summaries, func(i, j int) bool {
			return summaries[i].Task.ID > summaries[j].Task.ID
		})
	} else {
		sort.SliceStable(summaries, func(i, j int) bool {
			return summaries[i].Task.ID < summaries[j].Task.ID
		})
	}

	if !opts.showAll {
		window, err := time.ParseDuration(opts.durationRange)
		if err != nil {
			return fmt.Errorf("invalid duration %q: %w", opts.durationRange, err)
		}
		cutoff := time.Now().Add(-window)
		filtered := make([]TaskSummary, 0, len(summaries))
		for _, summary := range summaries {
			if ts := summary.effectiveStart(); ts != nil && ts.After(cutoff) {
				filtered = append(filtered, summary)
			}
		}
		summaries = filtered
	}

	if opts.limit > 0 && len(summaries) > opts.limit {
		summaries = summaries[:opts.limit]
	}

	if opts.jsonOutput {
		payload, err := buildJSONTasks(summaries)
		if err != nil {
			return err
		}
		if len(payload) == 0 {
			fmt.Println("[]")
			return nil
		}
		fmt.Println(string(payload))
		return nil
	}

	if len(summaries) == 0 {
		fmt.Println("No tasks found.")
		return nil
	}

	return renderTaskSummaries(summaries, ls.displayCfg)
}

func renderTaskSummaries(summaries []TaskSummary, cfg DisplayConfig) error {
	for _, summary := range summaries {
		fmt.Println(formatTaskSummary(summary, cfg))
	}
	return nil
}

func formatTaskSummary(summary TaskSummary, cfg DisplayConfig) string {
	startSegment := formatStartSegment(summary, cfg)
	durationSegment := formatDurationSegment(summary.Task.Duration)
	progressSegment := formatProgressSegment(summary)
	tagSegment := formatTagSegment(summary.Task.Tags, cfg.TagColors)
	return fmt.Sprintf("%d: %s %s %s%s - %s",
		summary.Task.ID, startSegment, durationSegment, progressSegment, tagSegment, summary.Task.Message)
}

func formatStartSegment(summary TaskSummary, cfg DisplayConfig) string {
	if summary.firstSessionStart == nil {
		return "[]"
	}
	format := cfg.DateTimeFormat
	if format == "" {
		format = defaultDateTimeFormat
	}
	return fmt.Sprintf("[%s]", summary.firstSessionStart.Format(format))
}

func formatDurationSegment(duration string) string {
	if duration == "" {
		return "[25m0s]"
	}
	if parsed, err := time.ParseDuration(duration); err == nil {
		return fmt.Sprintf("[%s]", parsed.Truncate(time.Second))
	}
	return fmt.Sprintf("[%s]", duration)
}

func formatDisplayTime(raw, dateFormat string) string {
	if raw == "" {
		return "N/A"
	}
	if ts := parseTimestamp(raw); ts != nil {
		format := dateFormat
		if format == "" {
			format = defaultDateTimeFormat
		}
		return ts.Format(format)
	}
	return raw
}

func formatProgressSegment(summary TaskSummary) string {
	var builder strings.Builder
	builder.WriteString("[")

	expectedSeconds := parseDurationToSeconds(summary.Task.Duration)
	progressCount := 0
	for i, session := range summary.Sessions {
		if i > 0 {
			builder.WriteString(" ")
		}
		builder.WriteString(colorizeSession(session, expectedSeconds))
		progressCount++
	}

	missing := summary.Task.Pomodoros - progressCount
	if missing < 0 {
		missing = 0
	}

	for i := 0; i < missing; i++ {
		if progressCount > 0 || i > 0 {
			builder.WriteString(" ")
		}
		builder.WriteString(color.New(color.FgRed).Sprint("X"))
	}

	builder.WriteString("]")
	return builder.String()
}

func formatTagSegment(tags []string, colorMap map[string]*color.Color) string {
	if len(tags) == 0 {
		return ""
	}
	var builder strings.Builder
	builder.WriteString(" [")
	for i, tag := range tags {
		if i > 0 {
			builder.WriteString(" ")
		}
		trimmed := strings.TrimSpace(tag)
		colorValue, ok := colorMap[trimmed]
		if !ok {
			colorValue, ok = colorMap[strings.ToLower(trimmed)]
		}
		if ok && colorValue != nil {
			builder.WriteString(colorValue.Sprint(trimmed))
		} else {
			builder.WriteString(trimmed)
		}
	}
	builder.WriteString("]")
	return builder.String()
}

func parseDurationToSeconds(duration string) int {
	if duration == "" {
		return 0
	}
	parsed, err := time.ParseDuration(duration)
	if err != nil {
		return 0
	}
	return int(parsed.Seconds())
}

func colorizeSession(session PomodoroSession, expectedSeconds int) string {
	green := color.New(color.FgGreen).SprintFunc()
	yellow := color.New(color.FgYellow).SprintFunc()
	red := color.New(color.FgRed).SprintFunc()

	if !session.Completed {
		return red("X")
	}

	actualSeconds := session.Duration
	if session.EndTime != "" && session.StartTime != "" {
		if start, err := time.Parse(time.RFC3339, session.StartTime); err == nil {
			if end, err := time.Parse(time.RFC3339, session.EndTime); err == nil {
				actualSeconds = int(end.Sub(start).Seconds())
			}
		}
	}

	if expectedSeconds > 0 && actualSeconds > expectedSeconds+300 {
		return yellow("X")
	}

	return green("X")
}

type jsonTask struct {
	ID         int            `json:"id"`
	Message    string         `json:"message"`
	Pomodoros  []jsonPomodoro `json:"pomodoros"`
	Tags       []string       `json:"tags"`
	NPomodoros int            `json:"n_pomodoros"`
	Duration   time.Duration  `json:"duration"`
}

type jsonPomodoro struct {
	Start *time.Time `json:"start"`
	End   *time.Time `json:"end"`
}

func buildJSONTasks(summaries []TaskSummary) ([]byte, error) {
	jsonTasks := make([]jsonTask, 0, len(summaries))
	for _, summary := range summaries {
		duration, err := time.ParseDuration(summary.Task.Duration)
		if err != nil {
			duration = 0
		}

		var tags []string
		if len(summary.Task.Tags) > 0 {
			tags = append([]string(nil), summary.Task.Tags...)
		}

		jTask := jsonTask{
			ID:         summary.Task.ID,
			Message:    summary.Task.Message,
			Tags:       tags,
			NPomodoros: summary.Task.Pomodoros,
			Duration:   duration,
			Pomodoros:  make([]jsonPomodoro, 0, len(summary.Sessions)),
		}

		for _, session := range summary.Sessions {
			jPomodoro := jsonPomodoro{}
			if start := parseTimestamp(session.StartTime); start != nil {
				jPomodoro.Start = start
			}
			if end := parseTimestamp(session.EndTime); end != nil {
				jPomodoro.End = end
			}
			jTask.Pomodoros = append(jTask.Pomodoros, jPomodoro)
		}

		jsonTasks = append(jsonTasks, jTask)
	}

	return json.Marshal(jsonTasks)
}

func splitArg(arg string) (string, string, bool) {
	if strings.Contains(arg, "=") {
		parts := strings.SplitN(arg, "=", 2)
		return parts[0], parts[1], true
	}
	return arg, "", false
}

func boolFlagValue(hasValue bool, value string, defaultWhenNoValue bool) (bool, error) {
	if !hasValue {
		return defaultWhenNoValue, nil
	}
	parsed, err := strconv.ParseBool(value)
	if err != nil {
		return false, fmt.Errorf("invalid boolean value %q", value)
	}
	return parsed, nil
}

func parseListArgs(args []string) (listOptions, error) {
	opts := defaultListOptions()

	for i := 0; i < len(args); i++ {
		raw := args[i]
		key, value, hasValue := splitArg(raw)

		switch key {
		case "--json":
			val, err := boolFlagValue(hasValue, value, true)
			if err != nil {
				return opts, err
			}
			opts.jsonOutput = val
		case "--no-json":
			opts.jsonOutput = false
		case "--assend":
			val, err := boolFlagValue(hasValue, value, true)
			if err != nil {
				return opts, err
			}
			opts.descending = val
		case "--no-assend":
			opts.descending = false
		case "-a", "--all":
			val, err := boolFlagValue(hasValue, value, true)
			if err != nil {
				return opts, err
			}
			opts.showAll = val
		case "--no-all":
			opts.showAll = false
		case "-n", "--limit":
			if !hasValue {
				if i+1 >= len(args) {
					return opts, fmt.Errorf("--limit requires a value")
				}
				value = args[i+1]
				hasValue = true
				i++
			}
			limit, err := strconv.Atoi(value)
			if err != nil {
				return opts, fmt.Errorf("invalid limit %q", value)
			}
			if limit < 0 {
				limit = 0
			}
			opts.limit = limit
		case "-d", "--duration":
			if !hasValue {
				if i+1 >= len(args) {
					return opts, fmt.Errorf("--duration requires a value")
				}
				value = args[i+1]
				hasValue = true
				i++
			}
			opts.durationRange = value
		case "--api-url":
			if !hasValue {
				if i+1 >= len(args) {
					return opts, fmt.Errorf("--api-url requires a value")
				}
				i++
			}
		default:
			return opts, fmt.Errorf("unknown list option %s", raw)
		}
	}

	return opts, nil
}

func (ls *LocalStore) getActiveSession() (*PomodoroSession, error) {
	db, err := ls.openDB()
	if err != nil {
		return nil, err
	}
	defer db.Close()

	row := db.QueryRow(
		`SELECT id, task_id, start_time, end_time, duration, completed, notes, created_at
		 FROM pomodoro_sessions
		 WHERE completed = FALSE
		 ORDER BY start_time DESC
		 LIMIT 1`,
	)

	var session PomodoroSession
	var endTime sql.NullString
	err = row.Scan(&session.ID, &session.TaskID, &session.StartTime, &endTime, &session.Duration, &session.Completed, &session.Notes, &session.CreatedAt)
	if err == sql.ErrNoRows {
		return nil, nil
	}
	if err != nil {
		return nil, err
	}
	if endTime.Valid {
		session.EndTime = endTime.String
	}
	return &session, nil
}

func (ls *LocalStore) currentStatus() (StatusInfo, error) {
	activeSession, err := ls.getActiveSession()
	if err != nil {
		return StatusInfo{}, err
	}
	if activeSession == nil {
		return StatusInfo{}, nil
	}

	status := StatusInfo{
		TaskID: activeSession.TaskID,
		State:  statusRunning,
	}

	task, err := ls.getTaskByID(activeSession.TaskID)
	if err != nil {
		return StatusInfo{}, err
	}
	if task != nil {
		status.TaskMessage = task.Message
		status.NPomodoros = task.Pomodoros
	}

	taskSessions, err := ls.getTaskSessions(activeSession.TaskID)
	if err != nil {
		return StatusInfo{}, err
	}
	for _, session := range taskSessions {
		if session.Completed {
			status.Count++
		}
	}

	if start := parseTimestamp(activeSession.StartTime); start != nil {
		now := time.Now()
		expected := time.Duration(activeSession.Duration) * time.Second
		if now.Before(*start) {
			status.Remaining = expected
			status.State = statusRunning
		} else {
			elapsed := now.Sub(*start)
			if elapsed < 0 {
				elapsed = 0
			}
			if elapsed >= expected {
				status.Remaining = 0
				status.State = statusBreaking
			} else {
				status.Remaining = expected - elapsed
				status.State = statusRunning
			}
		}
	} else {
		status.State = statusCreated
		status.Remaining = 0
	}

	status.PauseDuration = 0
	return status, nil
}

func formatStatusOutput(status StatusInfo) string {
	stateSymbol := "?"
	if status.State >= statusRunning {
		label := status.State.String()
		if label != "" {
			stateSymbol = string(label[0])
		}
	}

	if status.State == statusRunning {
		return fmt.Sprintf("%s [%d/%d] %s", stateSymbol, status.Count, status.NPomodoros, status.Remaining)
	}

	return fmt.Sprintf("%s [%d/%d] -", stateSymbol, status.Count, status.NPomodoros)
}

// Interactive Timer Implementation

func NewInteractiveTimer(store *LocalStore, session *PomodoroSession, task *Task, duration time.Duration) *InteractiveTimer {
	ctx, cancel := context.WithCancel(context.Background())
	return &InteractiveTimer{
		store:              store,
		session:            session,
		task:               task,
		duration:           duration,
		remaining:          duration,
		state:              statusRunning,
		ctx:                ctx,
		cancel:             cancel,
		breakDuration:      5 * time.Minute,
		completedPomodoros: 0,
	}
}

func (t *InteractiveTimer) Start() error {
	// Setup signal handling
	sigChan := make(chan os.Signal, 1)
	signal.Notify(sigChan, os.Interrupt, syscall.SIGTERM)

	// Setup keyboard input
	inputChan := make(chan rune, 10)
	go t.readInput(inputChan)

	// Start the timer loop
	ticker := time.NewTicker(100 * time.Millisecond)
	defer ticker.Stop()

	// Clear screen and hide cursor
	fmt.Print("\033[2J\033[?25l")
	defer fmt.Print("\033[?25h") // Show cursor on exit

	lastUpdate := time.Now()

	for {
		select {
		case <-t.ctx.Done():
			t.clearScreen()
			fmt.Println("\n✓ Timer stopped")
			return nil

		case <-sigChan:
			t.cancel()
			return nil

		case input := <-inputChan:
			t.handleInput(input)

		case <-ticker.C:
			if !t.paused && !t.inBreak {
				now := time.Now()
				elapsed := now.Sub(lastUpdate)
				lastUpdate = now

				t.mu.Lock()
				t.remaining -= elapsed
				if t.remaining <= 0 {
					t.remaining = 0
					t.mu.Unlock()
					if err := t.handleCompletion(); err != nil {
						return err
					}
					continue
				}
				t.mu.Unlock()
			} else if !t.paused && t.inBreak {
				now := time.Now()
				elapsed := now.Sub(lastUpdate)
				lastUpdate = now

				t.mu.Lock()
				t.remaining -= elapsed
				if t.remaining <= 0 {
					t.remaining = 0
					t.mu.Unlock()
					if err := t.endBreak(); err != nil {
						return err
					}
					return nil
				}
				t.mu.Unlock()
			}

			t.render()
		}
	}
}

func (t *InteractiveTimer) readInput(inputChan chan<- rune) {
	// Disable input buffering
	exec.Command("stty", "-F", "/dev/tty", "cbreak", "min", "1").Run()
	exec.Command("stty", "-F", "/dev/tty", "-echo").Run()
	defer exec.Command("stty", "-F", "/dev/tty", "echo").Run()

	buf := make([]byte, 1)
	for {
		select {
		case <-t.ctx.Done():
			return
		default:
			n, err := os.Stdin.Read(buf)
			if err != nil || n == 0 {
				time.Sleep(50 * time.Millisecond)
				continue
			}
			inputChan <- rune(buf[0])
		}
	}
}

func (t *InteractiveTimer) handleInput(input rune) {
	t.mu.Lock()
	defer t.mu.Unlock()

	switch input {
	case 'p', 'P', ' ': // Pause/Resume
		if t.paused {
			t.resume()
		} else {
			t.pause()
		}
	case 'q', 'Q', 3: // Quit (q, Q, or Ctrl+C)
		t.cancel()
	case 's', 'S': // Stop and complete
		t.remaining = 0
	}
}

func (t *InteractiveTimer) pause() {
	if !t.paused {
		t.paused = true
		t.pauseStart = time.Now()
		t.state = statusPaused
	}
}

func (t *InteractiveTimer) resume() {
	if t.paused {
		t.paused = false
		t.totalPaused += time.Since(t.pauseStart)
		t.state = statusRunning
	}
}

func (t *InteractiveTimer) handleCompletion() error {
	t.clearScreen()

	// Show completion message
	green := color.New(color.FgGreen, color.Bold)
	green.Println("\n✓ Pomodoro completed!")

	// Send notification
	t.sendNotification("Pomodoro Complete", "Time for a break!")

	// Mark session as complete
	if err := t.store.completeSession(t.session.ID, ""); err != nil {
		return err
	}

	t.completedPomodoros++

	// Determine break duration
	if t.completedPomodoros%4 == 0 {
		t.breakDuration = 15 * time.Minute
		fmt.Printf("\nTake a long break (15 minutes)\n")
	} else {
		t.breakDuration = 5 * time.Minute
		fmt.Printf("\nTake a short break (5 minutes)\n")
	}

	fmt.Println("\nPress any key to start break, or 'q' to quit...")

	// Wait for user input
	buf := make([]byte, 1)
	os.Stdin.Read(buf)
	if buf[0] == 'q' || buf[0] == 'Q' {
		t.cancel()
		return nil
	}

	// Start break
	return t.startBreak()
}

func (t *InteractiveTimer) startBreak() error {
	t.inBreak = true
	t.remaining = t.breakDuration
	t.state = statusBreaking
	t.paused = false

	t.clearScreen()
	yellow := color.New(color.FgYellow, color.Bold)
	yellow.Println("\n☕ Break time!")

	t.sendNotification("Break Started", fmt.Sprintf("Relax for %d minutes", int(t.breakDuration.Minutes())))

	return nil
}

func (t *InteractiveTimer) endBreak() error {
	t.clearScreen()
	green := color.New(color.FgGreen, color.Bold)
	green.Println("\n✓ Break complete!")

	t.sendNotification("Break Complete", "Ready for another pomodoro?")

	fmt.Println("\nPress any key to continue...")
	buf := make([]byte, 1)
	os.Stdin.Read(buf)

	return nil
}

func (t *InteractiveTimer) sendNotification(title, message string) {
	// Try to send desktop notification
	cmd := exec.Command("notify-send", title, message, "-u", "normal", "-t", "5000")
	cmd.Run() // Ignore errors if notify-send is not available
}

func (t *InteractiveTimer) clearScreen() {
	fmt.Print("\033[2J\033[H")
}

func (t *InteractiveTimer) render() {
	t.mu.Lock()
	defer t.mu.Unlock()

	// Move cursor to top
	fmt.Print("\033[H")

	// Create colored title
	cyan := color.New(color.FgCyan, color.Bold)
	yellow := color.New(color.FgYellow, color.Bold)
	green := color.New(color.FgGreen, color.Bold)
	red := color.New(color.FgRed, color.Bold)

	// Title
	fmt.Println()
	cyan.Println("  🍅 POMODORO TIMER")
	fmt.Println()

	// Task info
	if t.task != nil {
		fmt.Printf("  Task: %s\n", t.task.Message)
		if len(t.task.Tags) > 0 {
			fmt.Printf("  Tags: %s\n", strings.Join(t.task.Tags, ", "))
		}
		fmt.Println()
	}

	// Progress
	sessions, _ := t.store.getTaskSessions(t.task.ID)
	completed := 0
	for _, s := range sessions {
		if s.Completed {
			completed++
		}
	}
	fmt.Printf("  Progress: %d/%d pomodoros\n", completed, t.task.Pomodoros)
	fmt.Println()

	// Timer display
	minutes := int(t.remaining.Minutes())
	seconds := int(t.remaining.Seconds()) % 60

	timerStr := fmt.Sprintf("  %02d:%02d", minutes, seconds)

	if t.inBreak {
		yellow.Println(timerStr)
		yellow.Println("\n  ☕ BREAK TIME")
	} else if t.paused {
		red.Println(timerStr)
		red.Println("\n  ⏸  PAUSED")
	} else {
		green.Println(timerStr)
		green.Println("\n  ▶  RUNNING")
	}

	fmt.Println()

	// Progress bar
	t.renderProgressBar()

	fmt.Println()

	// Controls
	fmt.Println("  Controls:")
	fmt.Println("    [Space/P] Pause/Resume")
	fmt.Println("    [S]       Stop and complete")
	fmt.Println("    [Q]       Quit")
	fmt.Println()
}

func (t *InteractiveTimer) renderProgressBar() {
	barWidth := 40
	var progress float64
	if t.duration > 0 {
		progress = float64(t.duration-t.remaining) / float64(t.duration)
	}

	filled := int(progress * float64(barWidth))
	if filled > barWidth {
		filled = barWidth
	}
	if filled < 0 {
		filled = 0
	}

	green := color.New(color.FgGreen)
	gray := color.New(color.FgHiBlack)

	fmt.Print("  [")
	green.Print(strings.Repeat("█", filled))
	gray.Print(strings.Repeat("░", barWidth-filled))
	fmt.Print("]")
	fmt.Printf(" %.0f%%\n", progress*100)
}

func parseStatusArgs(args []string) (bool, error) {
	jsonOutput := false
	for i := 0; i < len(args); i++ {
		raw := args[i]
		key, value, hasValue := splitArg(raw)

		switch key {
		case "--json":
			val, err := boolFlagValue(hasValue, value, true)
			if err != nil {
				return false, err
			}
			jsonOutput = val
		case "--no-json":
			jsonOutput = false
		case "--api-url":
			if !hasValue {
				if i+1 >= len(args) {
					return false, fmt.Errorf("--api-url requires a value")
				}
				i++
			}
		default:
			return false, fmt.Errorf("unknown status option %s", raw)
		}
	}
	return jsonOutput, nil
}

func (ls *LocalStore) outputStatus(jsonOutput bool) error {
	status, err := ls.currentStatus()
	if err != nil {
		return err
	}

	if jsonOutput {
		payload, err := json.Marshal(status)
		if err != nil {
			return err
		}
		fmt.Println(string(payload))
		return nil
	}

	fmt.Println(formatStatusOutput(status))
	return nil
}

// API methods
func (ls *LocalStore) syncTasksToAPI() error {
	if ls.apiURL == "" {
		return fmt.Errorf("API URL not configured")
	}

	tasks, err := ls.getAllTasks()
	if err != nil {
		return err
	}

	for _, task := range tasks {
		payload := map[string]interface{}{
			"message":   task.Message,
			"pomodoros": task.Pomodoros,
			"duration":  task.Duration,
			"tags":      strings.Join(task.Tags, ","),
			"status":    task.Status,
		}

		taskJSON, err := json.Marshal(payload)
		if err != nil {
			continue
		}

		req, err := http.NewRequest("POST", ls.apiURL+"/api/tasks", bytes.NewBuffer(taskJSON))
		if err != nil {
			fmt.Printf("Error creating request for task %d: %v\n", task.ID, err)
			continue
		}
		req.Header.Set("Content-Type", "application/json")
		if ls.authToken != "" {
			req.Header.Set("Authorization", "Bearer "+ls.authToken)
		}

		client := &http.Client{}
		resp, err := client.Do(req)
		if err != nil {
			fmt.Printf("Error syncing task %d: %v\n", task.ID, err)
			continue
		}
		resp.Body.Close()

		if resp.StatusCode >= 200 && resp.StatusCode < 300 {
			fmt.Printf("Synced task %d to API\n", task.ID)
		} else {
			fmt.Printf("Failed to sync task %d. Status: %s\n", task.ID, resp.Status)
		}
	}

	return nil
}

func (ls *LocalStore) syncTasksFromAPI() error {
	if ls.apiURL == "" {
		return fmt.Errorf("API URL not configured")
	}

	req, err := http.NewRequest("GET", ls.apiURL+"/api/tasks", nil)
	if err != nil {
		return err
	}

	if ls.authToken != "" {
		req.Header.Set("Authorization", "Bearer "+ls.authToken)
	}

	client := &http.Client{}
	resp, err := client.Do(req)
	if err != nil {
		return err
	}
	defer resp.Body.Close()

	if resp.StatusCode != http.StatusOK {
		return fmt.Errorf("API returned status: %s", resp.Status)
	}

	body, err := io.ReadAll(resp.Body)
	if err != nil {
		return err
	}

	var rawTasks []map[string]interface{}
	if err := json.Unmarshal(body, &rawTasks); err != nil {
		return err
	}

	for _, raw := range rawTasks {
		var apiTask Task
		if msg, ok := raw["message"].(string); ok {
			apiTask.Message = msg
		}
		if p, ok := raw["pomodoros"].(float64); ok {
			apiTask.Pomodoros = int(p)
		}
		if dur, ok := raw["duration"].(string); ok {
			apiTask.Duration = dur
		}
		if status, ok := raw["status"].(string); ok {
			apiTask.Status = status
		}
		if created, ok := raw["created_at"].(string); ok {
			apiTask.CreatedAt = created
		}
		if updated, ok := raw["updated_at"].(string); ok {
			apiTask.UpdatedAt = updated
		}

		// tags may be string or array
		switch tags := raw["tags"].(type) {
		case string:
			if tags != "" {
				apiTask.Tags = strings.Split(tags, ",")
			}
		case []interface{}:
			for _, t := range tags {
				if s, ok := t.(string); ok && strings.TrimSpace(s) != "" {
					apiTask.Tags = append(apiTask.Tags, s)
				}
			}
		}

		// Check if task already exists locally
		existingTasks, err := ls.getAllTasks()
		if err != nil {
			continue
		}

		exists := false
		for _, existing := range existingTasks {
			if existing.Message == apiTask.Message && existing.CreatedAt == apiTask.CreatedAt {
				exists = true
				break
			}
		}

		if !exists {
			_, err := ls.createTask(apiTask)
			if err != nil {
				fmt.Printf("Error importing task %s: %v\n", apiTask.Message, err)
			} else {
				fmt.Printf("Imported task: %s\n", apiTask.Message)
			}
		}
	}

	return nil
}

// Pomo CLI integration
func (ls *LocalStore) runPomoCLI(args []string) error {
	if len(args) == 0 {
		return fmt.Errorf("no pomo command provided")
	}

	cmd := exec.Command("pomo", args...)
	cmd.Stdout = os.Stdout
	cmd.Stderr = os.Stderr
	cmd.Stdin = os.Stdin

	if err := cmd.Run(); err != nil {
		if errors.Is(err, exec.ErrNotFound) {
			fmt.Println("Original pomo binary not found, using integrated fallback.")
			return ls.runPomoFallback(args)
		}
		var exitErr *exec.ExitError
		if errors.As(err, &exitErr) {
			return fmt.Errorf("pomo command failed: %w", exitErr)
		}
		return err
	}

	return nil
}

func (ls *LocalStore) runPomoFallback(args []string) error {
	cmd := args[0]
	switch cmd {
	case "list":
		return ls.runPomoList()
	case "status":
		jsonOutput, err := parseStatusArgs(args[1:])
		if err != nil {
			return err
		}
		return ls.outputStatus(jsonOutput)
	case "start":
		if len(args) < 2 {
			return fmt.Errorf("task message required")
		}
		return ls.runPomoStart(strings.Join(args[1:], " "))
	default:
		return fmt.Errorf("unsupported pomo command: %s", cmd)
	}
}

func (ls *LocalStore) runPomoList() error {
	return ls.listTasks(defaultListOptions())
}

func (ls *LocalStore) runPomoStart(message string) error {
	// Create task locally and potentially sync with pomo CLI
	task := Task{
		Message:   message,
		Pomodoros: 1,
		Duration:  "25m",
		Tags:      []string{},
		Status:    "active",
	}

	id, err := ls.createTask(task)
	if err != nil {
		return err
	}

	fmt.Printf("Started task: %s (ID: %d)\n", message, id)
	fmt.Printf("This would normally start the pomo CLI timer\n")

	return nil
}

func parseCompleteArgs(args []string) (int, string, error) {
	if len(args) == 0 {
		return 0, "", fmt.Errorf("session ID is required")
	}

	sessionID := -1
	var notesParts []string

	for i := 0; i < len(args); i++ {
		arg := args[i]
		switch arg {
		case "--api-url":
			if i+1 >= len(args) {
				return 0, "", fmt.Errorf("--api-url requires a value")
			}
			i++
		case "-n", "--notes":
			if i+1 >= len(args) {
				return 0, "", fmt.Errorf("--notes requires a value")
			}
			notesParts = append(notesParts, args[i+1])
			i++
		default:
			if sessionID == -1 {
				if id, err := strconv.Atoi(arg); err == nil {
					sessionID = id
					continue
				}
			}

			if strings.HasPrefix(arg, "-") {
				return 0, "", fmt.Errorf("unknown flag %s", arg)
			}
			notesParts = append(notesParts, arg)
		}
	}

	if sessionID == -1 {
		return 0, "", fmt.Errorf("session ID is required")
	}

	notes := strings.Join(notesParts, " ")
	return sessionID, notes, nil
}

func printUsage() {
	fmt.Println("Pomo CLI with Local DB and API Sync")
	fmt.Println("\nUsage:")
	fmt.Println("  pomo-cli <command> [options]")
	fmt.Println("\nCommands:")
	fmt.Println("  init                    Initialize local database")
	fmt.Println("  add <message>           Add a new task")
	fmt.Println("  list                    List all tasks")
	fmt.Println("  status                  Show current pomodoro status")
	fmt.Println("  start <task-id>         Start a pomodoro session for a task")
	fmt.Println("  complete <session-id>   Complete a pomodoro session (use --notes to add notes)")
	fmt.Println("  sessions <task-id>      List sessions for a task")
	fmt.Println("  sync-to-api             Sync local data to API")
	fmt.Println("  sync-from-api           Sync data from API to local")
	fmt.Println("  pomo <command>          Run original pomo CLI command")
	fmt.Println("\nOptions:")
	fmt.Println("  -p, --pomodoros <n>     Number of pomodoros (default: 1)")
	fmt.Println("  -d, --duration <time>   Duration per pomodoro (default: 25m)")
	fmt.Println("  -t, --tags <tags>       Comma-separated tags")
	fmt.Println("  --api-url <url>         API server URL for syncing")
	fmt.Println("\nList options:")
	fmt.Println("  --json                  Output tasks as JSON")
	fmt.Println("  --assend                Show newest tasks first")
	fmt.Println("  --limit <n>             Limit number of tasks shown")
	fmt.Println("  --duration <window>     Limit tasks to recent window (use with --no-all)")
	fmt.Println("  --no-all                Apply time filtering window")
	fmt.Println("\nStatus options:")
	fmt.Println("  --json                  Output current status as JSON")
}

func main() {
	apiURL := os.Getenv("POMO_API_URL")
	if apiURL == "" {
		apiURL = "http://localhost:3000/api"
	}

	args := os.Args[1:]
	var command string
	var commandArgs []string

	for i := 0; i < len(args); i++ {
		arg := args[i]
		switch arg {
		case "--api-url":
			if i+1 >= len(args) {
				fmt.Println("Error: --api-url requires a value")
				os.Exit(1)
			}
			apiURL = args[i+1]
			i++
		case "-h", "--help":
			printUsage()
			return
		default:
			if command == "" {
				command = arg
			} else {
				commandArgs = append(commandArgs, arg)
			}
		}
	}

	if command == "" {
		printUsage()
		os.Exit(1)
	}

	store := NewLocalStore(apiURL)

	// Initialize database if needed
	if command != "init" {
		if err := store.initDB(); err != nil {
			log.Fatal("Error initializing database:", err)
		}
	}

	switch command {
	case "init":
		if err := store.initDB(); err != nil {
			log.Fatal("Error initializing database:", err)
		}
		fmt.Println("Local database initialized successfully at:", store.dbPath)

	case "add":
		cmdArgs := commandArgs
		if len(cmdArgs) == 0 {
			fmt.Println("Error: Message is required")
			fmt.Println("Usage: pomo-cli add <message> [options]")
			os.Exit(1)
		}

		pomodoros := 1
		duration := "25m"
		var tags []string
		var messageParts []string

		for i := 0; i < len(cmdArgs); i++ {
			arg := cmdArgs[i]

			switch arg {
			case "--api-url":
				if i+1 >= len(cmdArgs) {
					fmt.Println("Error: --api-url requires a value")
					os.Exit(1)
				}
				i++
				continue
			case "-p", "--pomodoros":
				if i+1 >= len(cmdArgs) {
					fmt.Println("Error: --pomodoros requires a value")
					os.Exit(1)
				}
				value := cmdArgs[i+1]
				i++
				p, err := strconv.Atoi(value)
				if err != nil {
					fmt.Printf("Error: invalid pomodoros value %q\n", value)
					os.Exit(1)
				}
				pomodoros = p
			case "-d", "--duration":
				if i+1 >= len(cmdArgs) {
					fmt.Println("Error: --duration requires a value")
					os.Exit(1)
				}
				duration = cmdArgs[i+1]
				i++
			case "-t", "--tags":
				if i+1 >= len(cmdArgs) {
					fmt.Println("Error: --tags requires a value")
					os.Exit(1)
				}
				rawTags := strings.Split(cmdArgs[i+1], ",")
				i++
				for _, tag := range rawTags {
					trimmed := strings.TrimSpace(tag)
					if trimmed != "" {
						tags = append(tags, trimmed)
					}
				}
			default:
				if strings.HasPrefix(arg, "-") {
					fmt.Printf("Error: unknown flag %s\n", arg)
					os.Exit(1)
				}
				messageParts = append(messageParts, arg)
			}
		}

		if len(messageParts) == 0 {
			fmt.Println("Error: Message is required")
			fmt.Println("Usage: pomo-cli add <message> [options]")
			os.Exit(1)
		}

		task := Task{
			Message:   strings.Join(messageParts, " "),
			Pomodoros: pomodoros,
			Duration:  duration,
			Tags:      tags,
			Status:    "active",
		}

		id, err := store.createTask(task)
		if err != nil {
			log.Fatal("Error creating task:", err)
		}
		fmt.Println(id)

	case "list":
		opts, err := parseListArgs(commandArgs)
		if err != nil {
			fmt.Println("Error:", err)
			os.Exit(1)
		}

		if err := store.listTasks(opts); err != nil {
			log.Fatal("Error listing tasks:", err)
		}

	case "status":
		jsonOutput, err := parseStatusArgs(commandArgs)
		if err != nil {
			fmt.Println("Error:", err)
			os.Exit(1)
		}

		if err := store.outputStatus(jsonOutput); err != nil {
			log.Fatal("Error retrieving status:", err)
		}

	case "start":
		if len(commandArgs) < 1 {
			fmt.Println("Error: Task ID is required")
			fmt.Println("Usage: pomo-cli start <task-id>")
			os.Exit(1)
		}

		taskID, err := strconv.Atoi(commandArgs[0])
		if err != nil {
			fmt.Println("Error: Invalid task ID")
			os.Exit(1)
		}

		// Get task to determine duration
		tasks, err := store.getAllTasks()
		if err != nil {
			log.Fatal("Error fetching tasks:", err)
		}

		var targetTask *Task
		for _, task := range tasks {
			if task.ID == taskID {
				targetTask = &task
				break
			}
		}

		if targetTask == nil {
			fmt.Printf("Task with ID %d not found\n", taskID)
			os.Exit(1)
		}

		// Parse duration
		duration, err := time.ParseDuration(targetTask.Duration)
		if err != nil {
			// Fallback to minutes if just a number
			if strings.HasSuffix(targetTask.Duration, "m") {
				duration, _ = time.ParseDuration(targetTask.Duration)
			} else {
				duration = 25 * time.Minute
			}
		}

		session := PomodoroSession{
			TaskID:    taskID,
			StartTime: time.Now().Format(time.RFC3339),
			Duration:  int(duration.Seconds()),
			Completed: false,
		}

		sessionID, err := store.createSession(session)
		if err != nil {
			log.Fatal("Error creating session:", err)
		}

		session.ID = sessionID

		// Start interactive timer
		timer := NewInteractiveTimer(store, &session, targetTask, duration)
		if err := timer.Start(); err != nil {
			log.Fatal("Error running timer:", err)
		}

	case "complete":
		sessionID, notes, err := parseCompleteArgs(commandArgs)
		if err != nil {
			fmt.Println("Error:", err)
			fmt.Println("Usage: pomo-cli complete <session-id> [--notes \"text\"]")
			os.Exit(1)
		}

		if err := store.completeSession(sessionID, notes); err != nil {
			log.Fatal("Error completing session:", err)
		}

		fmt.Printf("Session %d marked as complete", sessionID)
		if notes != "" {
			fmt.Printf(" (notes: %s)", notes)
		}
		fmt.Println()

	case "sessions":
		if len(commandArgs) < 1 {
			fmt.Println("Error: Task ID is required")
			fmt.Println("Usage: pomo-cli sessions <task-id>")
			os.Exit(1)
		}

		taskID, err := strconv.Atoi(commandArgs[0])
		if err != nil {
			fmt.Println("Error: Invalid task ID")
			os.Exit(1)
		}

		sessions, err := store.getTaskSessions(taskID)
		if err != nil {
			log.Fatal("Error fetching sessions:", err)
		}

		if len(sessions) == 0 {
			fmt.Printf("No sessions found for task %d\n", taskID)
			return
		}

		fmt.Printf("%-8s %-8s %-19s %-19s %-10s\n", "ID", "Task", "Start", "End", "Completed")
		fmt.Println(strings.Repeat("-", 72))
		for _, session := range sessions {
			completed := "No"
			if session.Completed {
				completed = "Yes"
			}
			fmt.Printf("%-8d %-8d %-19s %-19s %-10s\n",
				session.ID,
				session.TaskID,
				formatDisplayTime(session.StartTime, store.displayCfg.DateTimeFormat),
				formatDisplayTime(session.EndTime, store.displayCfg.DateTimeFormat),
				completed,
			)
		}

	case "sync-to-api":
		fmt.Println("Syncing local tasks to API...")
		if err := store.syncTasksToAPI(); err != nil {
			fmt.Printf("Error syncing to API: %v\n", err)
		} else {
			fmt.Println("Sync to API completed successfully")
		}

	case "sync-from-api":
		fmt.Println("Syncing tasks from API...")
		if err := store.syncTasksFromAPI(); err != nil {
			fmt.Printf("Error syncing from API: %v\n", err)
		} else {
			fmt.Println("Sync from API completed successfully")
		}

	case "pomo":
		if len(commandArgs) < 1 {
			fmt.Println("Error: Pomo command required")
			fmt.Println("Usage: pomo-cli pomo <command> [args...]")
			os.Exit(1)
		}

		if err := store.runPomoCLI(commandArgs); err != nil {
			log.Fatal("Error running pomo command:", err)
		}

	default:
		fmt.Printf("Unknown command: %s\n", command)
		printUsage()
		os.Exit(1)
	}
}
