const { Pool } = require('pg');

// Migration script to create the database schema
const migrateDatabase = async () => {
  const pool = new Pool({
    connectionString: process.env.DATABASE_URL,
  });

  try {
    console.log('Starting database migration...');

    // Create users table
    await pool.query(`
      CREATE TABLE IF NOT EXISTS users (
        id SERIAL PRIMARY KEY,
        username VARCHAR(50) UNIQUE NOT NULL,
        email VARCHAR(100) UNIQUE NOT NULL,
        password_hash VARCHAR(255) NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      );
    `);
    console.log('✓ Users table created');

    // Create tasks table
    await pool.query(`
      CREATE TABLE IF NOT EXISTS tasks (
        id SERIAL PRIMARY KEY,
        user_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
        message TEXT NOT NULL,
        pomodoros INTEGER DEFAULT 1,
        duration VARCHAR(20) DEFAULT '25m',
        tags TEXT DEFAULT '',
        status VARCHAR(20) DEFAULT 'active',
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      );
    `);
    console.log('✓ Tasks table created');

    // Create pomodoro_sessions table
    await pool.query(`
      CREATE TABLE IF NOT EXISTS pomodoro_sessions (
        id SERIAL PRIMARY KEY,
        task_id INTEGER REFERENCES tasks(id) ON DELETE CASCADE,
        start_time TIMESTAMP NOT NULL,
        end_time TIMESTAMP,
        duration INTEGER NOT NULL, -- in minutes
        completed BOOLEAN DEFAULT FALSE,
        notes TEXT,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      );
    `);
    console.log('✓ Pomodoro sessions table created');

    // Create indexes for better performance
    await pool.query('CREATE INDEX IF NOT EXISTS idx_tasks_user_id ON tasks(user_id);');
    await pool.query('CREATE INDEX IF NOT EXISTS idx_tasks_status ON tasks(status);');
    await pool.query('CREATE INDEX IF NOT EXISTS idx_sessions_task_id ON pomodoro_sessions(task_id);');
    await pool.query('CREATE INDEX IF NOT EXISTS idx_sessions_completed ON pomodoro_sessions(completed);');
    console.log('✓ Indexes created');

    console.log('Database migration completed successfully!');
  } catch (error) {
    console.error('Migration failed:', error);
    throw error;
  } finally {
    await pool.end();
  }
};

// Data migration script to import existing SQLite data
const importSQLiteData = async () => {
  const sqlite3 = require('sqlite3').verbose();
  const pool = new Pool({
    connectionString: process.env.DATABASE_URL,
  });

  const sqliteDb = new sqlite3.Database('/home/shantanu/.local/share/pomo/pomo.db');

  try {
    console.log('Starting data migration from SQLite...');

    // Create a default user (you'll want to add proper user management later)
    const defaultUser = await pool.query(
      'INSERT INTO users (username, email, password_hash) VALUES ($1, $2, $3) RETURNING id',
      ['default_user', 'user@example.com', 'temporary_hash']
    );
    const userId = defaultUser.rows[0].id;

    // Import tasks from SQLite
    const sqliteTasks = await new Promise((resolve, reject) => {
      sqliteDb.all('SELECT * FROM task', (err, rows) => {
        if (err) reject(err);
        else resolve(rows);
      });
    });

    for (const task of sqliteTasks) {
      await pool.query(
        'INSERT INTO tasks (user_id, message, pomodoros, duration, tags) VALUES ($1, $2, $3, $4, $5)',
        [userId, task.message, task.pomodoros, task.duration, task.tags]
      );
    }

    console.log(`✓ Imported ${sqliteTasks.length} tasks from SQLite`);

    // Import pomodoro sessions from SQLite
    const sqlitePomodoros = await new Promise((resolve, reject) => {
      sqliteDb.all('SELECT * FROM pomodoro', (err, rows) => {
        if (err) reject(err);
        else resolve(rows);
      });
    });

    for (const pomodoro of sqlitePomodoros) {
      // Find the corresponding task (this is a simple approach - you might need better mapping)
      const taskResult = await pool.query(
        'SELECT id FROM tasks WHERE message = $1 ORDER BY id DESC LIMIT 1',
        [/* You'll need to map this properly */]
      );

      if (taskResult.rows.length > 0) {
        await pool.query(
          'INSERT INTO pomodoro_sessions (task_id, start_time, end_time, duration, completed) VALUES ($1, $2, $3, $4, $5)',
          [
            taskResult.rows[0].id,
            pomodoro.start,
            pomodoro.end,
            25, // default 25 minutes
            pomodoro.end ? true : false
          ]
        );
      }
    }

    console.log(`✓ Imported ${sqlitePomodoros.length} pomodoro sessions from SQLite`);

  } catch (error) {
    console.error('Data migration failed:', error);
    throw error;
  } finally {
    sqliteDb.close();
    await pool.end();
  }
};

// Run migrations if this script is executed directly
if (require.main === module) {
  require('dotenv').config();
  
  migrateDatabase()
    .then(() => {
      console.log('Migration complete! You can now start the server.');
      process.exit(0);
    })
    .catch((error) => {
      console.error('Migration failed:', error);
      process.exit(1);
    });
}

module.exports = { migrateDatabase, importSQLiteData };